package ru.yandex.intranet.imscore.infrastructure.data.repositories.identityRelation.jpa

import org.springframework.data.jpa.repository.JpaRepository
import org.springframework.data.jpa.repository.JpaSpecificationExecutor
import org.springframework.data.jpa.repository.Modifying
import org.springframework.data.jpa.repository.Query
import org.springframework.stereotype.Repository
import ru.yandex.intranet.imscore.infrastructure.data.entities.identityRelation.IdentityRelationEntity
import ru.yandex.intranet.imscore.infrastructure.data.entities.identityRelation.IdentityRelationEntityId
import java.util.UUID

/**
 * Identity Relation jpa repository interface
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Repository
interface JpaIdentityRelationRepository: JpaRepository<IdentityRelationEntity, IdentityRelationEntityId>,
    JpaSpecificationExecutor<IdentityRelationEntity> {

    @Query(value = "WITH RECURSIVE r AS (" +
        "SELECT i.identity_id, i.group_id, i.connection_type from identity_relation i " +
            "WHERE i.identity_id = :identityId " +
            "AND (" +
                ":#{#connectionType.name()} = 'UNDEFINED' " +
                "OR i.connection_type = cast(:#{#connectionType.name()} as identity_connection_type)" +
            ") " +
        "UNION " +
        "SELECT ir.identity_id, ir.group_id, ir.connection_type from identity_relation ir " +
            "JOIN r ON ir.identity_id = r.group_id " +
            "AND (:onlyDirectly is false) " +
            "AND (" +
                ":#{#connectionType.name()} = 'UNDEFINED' " +
                "OR ir.connection_type = cast(:#{#connectionType.name()} as identity_connection_type)" +
            ") " +
        ") SELECT * FROM r where (:cursor IS NULL OR r.group_id > cast(cast(:cursor as text) as uuid)) limit :limit",
        nativeQuery = true)
    fun findAllGroupsByIdentityId(
        identityId: UUID,
        connectionType: IdentityRelationEntity.ConnectionType,
        onlyDirectly: Boolean,
        cursor: UUID?, limit: Int = 20
    ): List<IdentityRelationEntity>

    @Query(value = "WITH RECURSIVE r AS (" +
        "SELECT i.identity_id, i.group_id, i.connection_type from identity_relation i " +
            "WHERE i.identity_id = :identityId " +
            "AND (" +
                "cast(:#{#connectionType.name()} as identity_connection_type) = 'UNDEFINED' " +
                "OR i.connection_type = cast(:#{#connectionType.name()} as identity_connection_type)" +
            ") " +
        "UNION " +
        "SELECT ir.identity_id, ir.group_id, ir.connection_type from identity_relation ir " +
            "JOIN r ON ir.identity_id = r.group_id " +
            "AND (:onlyDirectly is false) " +
            "AND (" +
                "cast(:#{#connectionType.name()} as identity_connection_type) = 'UNDEFINED' " +
                "OR ir.connection_type = cast(:#{#connectionType.name()} as identity_connection_type)" +
            ") " +
        ") SELECT count(*) > 0 FROM r where group_id = :groupId",
        nativeQuery = true)
    fun existsIdentityRelationToGroup(
        identityId: UUID,
        groupId: UUID,
        connectionType: IdentityRelationEntity.ConnectionType,
        onlyDirectly: Boolean): Boolean

    @Modifying
    @Query("DELETE from IdentityRelationEntity i " +
        "where i.identityId IN (:identityIds) AND i.groupId = :groupId " +
        "AND i.connectionType = :connectionType")
    fun deleteByIdentityIdInAndGroupIdAndConnectionType(
        identityIds: List<UUID>, groupId: UUID, connectionType: IdentityRelationEntity.ConnectionType)

    @Modifying
    @Query("""DELETE FROM IdentityRelationEntity i
        WHERE i.identityId IN (:identityIds) AND i.groupId = :groupId""")
    fun deleteByIdentityIdInAndGroupId(
        identityIds: List<UUID>, groupId: UUID)

    @Modifying
    @Query(value = "INSERT INTO identity_relation(identity_id, group_id, connection_type) " +
        "VALUES(:identityId, :groupId, cast(:#{#connectionType.name()} as identity_connection_type)) " +
        "ON CONFLICT (identity_id, group_id) " +
        "DO UPDATE SET connection_type = cast(:#{#connectionType.name()} as identity_connection_type) " +
        "WHERE identity_relation.connection_type = 'MEMBERSHIP'",
        nativeQuery = true)
    fun upsert(identityId: UUID, groupId: UUID, connectionType: IdentityRelationEntity.ConnectionType)

    @Query(value = """
        SELECT * FROM identity_relation
         WHERE group_id = :groupId
    """, nativeQuery = true)
    fun findAllDirectIdentityRelationByGroupId(groupId: UUID): List<IdentityRelationEntity>
}
