package ru.yandex.intranet.imscore.infrastructure.data.repositories.identityType

import org.springframework.data.domain.PageRequest
import org.springframework.data.domain.Sort
import org.springframework.stereotype.Repository
import ru.yandex.intranet.imscore.core.domain.IdentityTypeSource.IdentityTypeSource
import ru.yandex.intranet.imscore.core.domain.identityType.IdentityType
import ru.yandex.intranet.imscore.core.domain.identityType.specifications.IdentityTypeListSpecification
import ru.yandex.intranet.imscore.core.exceptions.identityType.IdentityTypeNotFoundException
import ru.yandex.intranet.imscore.core.ports.identityType.IdentityTypeRepository
import ru.yandex.intranet.imscore.infrastructure.data.converters.identityType.IdentityTypeConverters
import ru.yandex.intranet.imscore.infrastructure.data.converters.identityTypeSource.IdentityTypeSourceConverters
import ru.yandex.intranet.imscore.infrastructure.data.repositories.identityType.jpa.JpaIdentityTypeRepository
import ru.yandex.intranet.imscore.infrastructure.data.repositories.identityType.specifications.IdentityTypeSpecifications

/**
 * IdentityType repository implementation
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Repository
open class IdentityTypeRepositoryImpl(
    private val jpaIdentityTypeRepository: JpaIdentityTypeRepository,
): IdentityTypeRepository {

    override fun getListBySpec(spec: IdentityTypeListSpecification): List<IdentityType> {
        val jpaSpec = IdentityTypeSpecifications.greaterId(spec.cursor)
        val pageable = PageRequest.of(0, spec.size, Sort.Direction.ASC, "id")
        val page = jpaIdentityTypeRepository.findAll(jpaSpec, pageable)
        return IdentityTypeConverters.toIdentityTypeList(page)
    }

    override fun save(model: IdentityType, identityTypeSource: IdentityTypeSource): IdentityType {
        val identityTypeEntity = IdentityTypeConverters.toIdentityTypeEntity(model,
            IdentityTypeSourceConverters.toIdentityTypeSourceEntity(identityTypeSource))
        jpaIdentityTypeRepository.saveAndFlush(identityTypeEntity)

        return IdentityTypeConverters.toIdentityType(identityTypeEntity)
    }

    override fun getById(id: String): IdentityType {
        return IdentityTypeConverters.toIdentityType(jpaIdentityTypeRepository.findById(id).orElseThrow {
            IdentityTypeNotFoundException(id)
        })
    }

    override fun findById(id: String): IdentityType? {
        val optional = jpaIdentityTypeRepository.findById(id)
        return optional
            .map { IdentityTypeConverters.toIdentityType(it) }
            .orElse(null)
    }

    override fun findBySourceId(sourceId: String): List<IdentityType> {
        val entityList = jpaIdentityTypeRepository.findBySourceId(sourceId)
        return entityList
            .map { IdentityTypeConverters.toIdentityType(it) }
    }
}
