package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.converters.identity

import com.google.protobuf.FieldMask
import com.google.protobuf.util.FieldMaskUtil
import com.google.protobuf.util.Timestamps.fromMillis
import ru.yandex.intranet.imscore.core.domain.identity.Identity
import ru.yandex.intranet.imscore.core.domain.identity.IdentityExternalId
import ru.yandex.intranet.imscore.core.domain.identity.IdentityIdOneOf
import ru.yandex.intranet.imscore.core.domain.identity.specification.CreateIdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.DeleteIdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.IdentityListSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.IdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.ModifiableIdentityDataSpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.MoveIdentitySpecification
import ru.yandex.intranet.imscore.core.domain.identity.specification.UpdateIdentitySpecification
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.defaultValue
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.PageTokenUtils.encode
import ru.yandex.intranet.imscore.infrastructure.shared.Constants.Companion.DEFAULT_PAGE_SIZE
import ru.yandex.intranet.imscore.proto.identity.CreateIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.CreateIdentityResponse
import ru.yandex.intranet.imscore.proto.identity.DeleteIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.GetIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.GetIdentityResponse
import ru.yandex.intranet.imscore.proto.identity.IdentityCompositeId
import ru.yandex.intranet.imscore.proto.identity.IdentityData
import ru.yandex.intranet.imscore.proto.identity.ListIdentitiesRequest
import ru.yandex.intranet.imscore.proto.identity.ListIdentitiesResponse
import ru.yandex.intranet.imscore.proto.identity.ModifiableIdentityData
import ru.yandex.intranet.imscore.proto.identity.MoveIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.UpdateIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.UpdateIdentityResponse
import ru.yandex.intranet.imscore.proto.identityType.IdentityType
import java.util.Optional
import java.util.UUID
import java.util.function.Consumer

/**
 * Identity proto converters
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
class IdentityProtoConverters {
    companion object {
        private const val IDENTITY_DATA_FIELD_NUMBER = 5

        private val IDENTITY_DATA_FIELD_NAME: String =
            ru.yandex.intranet.imscore.proto.identity.Identity.getDescriptor()
                .findFieldByNumber(IDENTITY_DATA_FIELD_NUMBER).name

        fun toIdentityListSpecification(from: ListIdentitiesRequest, cursor: UUID?):
            IdentityListSpecification {
            val canonicalFieldMask = FieldMaskUtil.normalize(from.fieldMask)
            val loadData: Boolean =  canonicalFieldMask.pathsList.stream()
                .anyMatch(IDENTITY_DATA_FIELD_NAME::equals)

            return IdentityListSpecification(
                cursor = cursor,
                size = if(from.pageSize > 0) from.pageSize else DEFAULT_PAGE_SIZE,
                loadData = loadData,
                groupId = if (from.groupId != defaultValue) UUID.fromString(from.groupId) else null,
                onlyDirectly = from.onlyDirectly
            )
        }

        fun toListIdentitiesResponse(
            from: List<Identity>, pageSize: Int, fieldMask: FieldMask): ListIdentitiesResponse {
            val identities = from.stream().map {
                toIdentityProto(it, fieldMask)
            }.toList()
            val list = ListIdentitiesResponse.newBuilder()
                .addAllIdentities(identities)

            if (from.isNotEmpty() && from.size == pageSize) {
                list.nextPageToken = encode(
                    listOf(from.last().id
                        .toString()))
            }
            return list.build()
        }

        fun toIdentitySpecification (from: GetIdentityRequest): IdentitySpecification {
            val canonicalFieldMask = FieldMaskUtil.normalize(from.fieldMask)
            val withData: Boolean =  canonicalFieldMask.pathsList.stream()
                .anyMatch(IDENTITY_DATA_FIELD_NAME::equals)

            val spec = if (from.identity.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                IdentitySpecification(
                    idOneOf = IdentityIdOneOf(UUID.fromString(from.identity.id), null),
                    withData = withData
                )
            } else {
                val externalIdentity = from.identity.externalIdentity
                IdentitySpecification(
                    idOneOf = IdentityIdOneOf(null,
                        IdentityExternalId(externalIdentity.externalId, externalIdentity.typeId)),
                    withData = withData
                )
            }
            return spec
        }

        fun toGetIdentityResponse(from: Identity, fieldMask: FieldMask): GetIdentityResponse {
            return GetIdentityResponse.newBuilder().setIdentity(
                toIdentityProto(from, fieldMask)
            ).build()
        }

        fun toCreateIdentitySpecification(from: CreateIdentityRequest): CreateIdentitySpecification {
            val externalId =
                if (from.externalIdentity.externalId == defaultValue) {
                    null
                } else  {
                    from.externalIdentity.externalId
                }

            val data = if (from.hasData()) {
                toIdentityData(from.data)
            } else {
                null
            }

            val identityIdOneofCase = from.parentId.identityIdOneofCase
            val parentId = if (identityIdOneofCase == null
                || identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.IDENTITYIDONEOF_NOT_SET) {
                null
            } else {
                if (identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    IdentityIdOneOf(UUID.fromString(from.parentId.id), null)
                } else {
                    val externalIdentity = from.parentId.externalIdentity
                    IdentityIdOneOf(null, IdentityExternalId(externalIdentity.externalId, externalIdentity.typeId))
                }
            }

            return CreateIdentitySpecification(
                IdentityExternalId(
                    externalId,
                    from.externalIdentity.typeId
                ), parentId, data
            )
        }

        fun toCreateIdentityResponse(from: Identity): CreateIdentityResponse {
            return CreateIdentityResponse.newBuilder()
                .setIdentity(toIdentityProto(from, null))
                .build()
        }

        fun toUpdateSpecification(from: UpdateIdentityRequest): UpdateIdentitySpecification {
            val data = if (from.hasData()) {
                toIdentityData(from.data)
            } else {
                null
            }

            return if (from.identity.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                UpdateIdentitySpecification(
                    UUID.fromString(
                        from.identity.id
                    ), data
                )
            } else {
                UpdateIdentitySpecification(
                    IdentityExternalId(
                        from.identity.externalIdentity.externalId,
                        from.identity.externalIdentity.typeId
                    ), data
                )
            }
        }

        fun toUpdateIdentityResponse(from: Identity): UpdateIdentityResponse {
            return UpdateIdentityResponse.newBuilder()
                .setIdentity(toIdentityProto(from, null))
                .build()
        }

        private fun toIdentityData(from: ModifiableIdentityData): ModifiableIdentityDataSpecification {
            return ModifiableIdentityDataSpecification(
                toNullableStringOptional(from.hasSlug(), from.slug.value),
                toNullableStringOptional(from.hasName(), from.name.value),
                toNullableStringOptional(from.hasLastname(), from.lastname.value),
                toNullableStringOptional(from.hasPhone(), from.phone.value),
                toNullableStringOptional(from.hasEmail(), from.email.value),
                toNullableStringOptional(from.hasAdditionalData(), from.additionalData.value),
            )
        }

        private fun toNullableStringOptional(has: Boolean, value: String): Optional<String>? {
            return if (has) {
                if (value == defaultValue) {
                    Optional.empty<String>()
                } else {
                    Optional.of(value)
                }
            } else {
                null
            }
        }

        fun toIdentityProto(from: Identity, fieldMask: FieldMask?):
            ru.yandex.intranet.imscore.proto.identity.Identity {
            val identityMessageBuilder = ru.yandex.intranet.imscore.proto.identity.Identity.newBuilder()
            if (from.externalId != null) {
                identityMessageBuilder.externalId = from.externalId
            }
            identityMessageBuilder.type = IdentityType.newBuilder()
                .setId(from.type.id)
                .setIsGroup(from.type.isGroup)
                .setSourceId(from.type.sourceId)
                .build()

            if (from.data != null) {
                val identityDataBuilder = IdentityData.newBuilder()
                setDataField(from.data.slug) { identityDataBuilder.slug = it }
                setDataField(from.data.name) { identityDataBuilder.name = it }
                setDataField(from.data.lastname) { identityDataBuilder.lastname = it }
                setDataField(from.data.email) { identityDataBuilder.email = it }
                setDataField(from.data.phone) { identityDataBuilder.phone = it }
                setDataField(from.data.additionalData) { identityDataBuilder.additionalData = it }
                identityMessageBuilder.data = identityDataBuilder.build()
            }

            if (from.parentId != null) {
                identityMessageBuilder.parentId = from.parentId.toString()
            }

            val identityMessage = identityMessageBuilder
                .setId(from.id.toString())
                .setCreatedAt(fromMillis(from.createdAt!!.toEpochMilli()))
                .setModifiedAt(fromMillis(from.modifiedAt!!.toEpochMilli()))
                .build()

            if (fieldMask == null || fieldMask == FieldMask.getDefaultInstance()) {
                return identityMessage
            }
            val identityBuilder = ru.yandex.intranet.imscore.proto.identity.Identity.newBuilder()
            FieldMaskUtil.merge(fieldMask, identityMessage, identityBuilder)
            return identityBuilder.build()
        }

        private fun setDataField(value: String?, setter: Consumer<String>) {
            if (value != null) {
                setter.accept(value)
            }
        }

        fun toDeleteIdentitySpecifications(from: DeleteIdentityRequest): DeleteIdentitySpecification {
            return if (from.identity.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID)
                DeleteIdentitySpecification(UUID.fromString(from.identity.id))
            else DeleteIdentitySpecification(from.identity.externalIdentity.externalId,
                from.identity.externalIdentity.typeId)
        }

        fun toMoveIdentitySpecification(request: MoveIdentityRequest): MoveIdentitySpecification {
            return MoveIdentitySpecification(
                identity = if (request.identity.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    IdentityIdOneOf(
                        UUID.fromString(request.identity.id),
                    )
                } else {
                    IdentityIdOneOf(
                        IdentityExternalId(
                            request.identity.externalIdentity.externalId,
                            request.identity.externalIdentity.typeId,
                        ),
                    )
                },
                toGroup = if (request.toGroup.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    IdentityIdOneOf(
                        UUID.fromString(request.toGroup.id),
                    )
                } else {
                    IdentityIdOneOf(
                        IdentityExternalId(
                            request.toGroup.externalIdentity.externalId,
                            request.toGroup.externalIdentity.typeId,
                        )
                    )
                }
            )
        }
    }
}
