package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.converters.identityGroup


import ru.yandex.intranet.imscore.core.domain.identity.IdentityExternalId
import ru.yandex.intranet.imscore.core.domain.identity.IdentityIdOneOf
import ru.yandex.intranet.imscore.core.domain.identityRelation.IdentityRelation
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.AddToGroupSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.ExistsInGroupSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.ListIdentityGroupsSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.RemoveFromGroupSpecification
import ru.yandex.intranet.imscore.core.domain.identityRelation.specifications.group.ReplaceGroupSpecification
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.converters.identity.IdentityProtoConverters
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.PageTokenUtils
import ru.yandex.intranet.imscore.infrastructure.shared.Constants
import ru.yandex.intranet.imscore.proto.identity.IdentityCompositeId
import ru.yandex.intranet.imscore.proto.identityGroup.AddToGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ExistsInGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ExistsInGroupResponse
import ru.yandex.intranet.imscore.proto.identityGroup.ListIdentityGroupsRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ListIdentityGroupsResponse
import ru.yandex.intranet.imscore.proto.identityGroup.RemoveFromGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ReplaceGroupRequest
import java.util.UUID

/**
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
class IdentityGroupProtoConverters {

    companion object {

        fun toListIdentityGroupsSpecification(
            request: ListIdentityGroupsRequest,
            cursor: UUID?
        ): ListIdentityGroupsSpecification {
            return ListIdentityGroupsSpecification(
                cursor = cursor,
                size = if(request.pageSize > 0) request.pageSize else Constants.DEFAULT_PAGE_SIZE,
                onlyDirectly = request.onlyDirectly,
                identity = if (request.identity.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    IdentityIdOneOf(
                        UUID.fromString(request.identity.id)
                    )
                } else {
                    IdentityIdOneOf(
                        IdentityExternalId(
                            request.identity.externalIdentity.externalId,
                            request.identity.externalIdentity.typeId,
                        )
                    )
                },
            )
        }

        fun toListIdentityGroupsResponse(
            from: List<IdentityRelation>, pageSize: Int,
        ): ListIdentityGroupsResponse {
            val identityRelations = from.stream().map {
                IdentityProtoConverters.toIdentityProto(it.group!!, null)
            }.toList()
            val list = ListIdentityGroupsResponse.newBuilder().addAllIdentities(identityRelations)

            if (from.isNotEmpty() && from.size == pageSize) {
                list.nextPageToken = PageTokenUtils.encode(
                    listOf(
                        from.last().groupId
                            .toString()
                    )
                )
            }
            return list.build()
        }

        fun toIdentityRelationToGroupSpecification(
            request: ExistsInGroupRequest
        ): ExistsInGroupSpecification {
            return ExistsInGroupSpecification(
                onlyDirectly = request.onlyDirectly,
                group = if (request.group.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    IdentityIdOneOf(
                        UUID.fromString(request.group.id)
                    )
                } else {
                    IdentityIdOneOf(
                        IdentityExternalId(
                            request.group.externalIdentity.externalId,
                            request.group.externalIdentity.typeId,
                        )
                    )
                },
                identity = if (request.identity.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    IdentityIdOneOf(
                        UUID.fromString(request.identity.id)
                    )
                } else {
                    IdentityIdOneOf(
                        IdentityExternalId(
                            request.identity.externalIdentity.externalId,
                            request.identity.externalIdentity.typeId,
                        )
                    )
                },
            )
        }

        fun toExistsInGroupResponse(exists: Boolean): ExistsInGroupResponse {
            return ExistsInGroupResponse.newBuilder().setExists(exists).build()
        }

        fun toAddIdentityRelationsToGroupSpecification(
            request: AddToGroupRequest): AddToGroupSpecification {

            val identities = toIdentities(request.identitiesList)

            return AddToGroupSpecification(
                group = if (request.group.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    IdentityIdOneOf(
                        UUID.fromString(request.group.id)
                    )
                } else {
                    IdentityIdOneOf(
                        IdentityExternalId(
                            request.group.externalIdentity.externalId,
                            request.group.externalIdentity.typeId,
                        )
                    )
                },
                identities = identities
            )
        }

        private fun toIdentities(identityCompositeIdList: List<IdentityCompositeId>): ArrayList<IdentityIdOneOf> {
            val identities = ArrayList<IdentityIdOneOf>(identityCompositeIdList.size)
            identityCompositeIdList.forEach {
                identities.add(
                    if (it.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                        IdentityIdOneOf(
                            UUID.fromString(it.id),
                        )
                    } else {
                        IdentityIdOneOf(
                            IdentityExternalId(
                                it.externalIdentity.externalId,
                                it.externalIdentity.typeId
                            )
                        )
                    }
                )
            }
            return identities
        }

        fun toRemoveIdentityRelationsToGroupSpecification(
            request: RemoveFromGroupRequest): RemoveFromGroupSpecification {

            val identities = toIdentities(request.identitiesList)

            return RemoveFromGroupSpecification(
                group = if (request.group.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    IdentityIdOneOf(
                        UUID.fromString(request.group.id)
                    )
                } else {
                    IdentityIdOneOf(
                        IdentityExternalId(
                            request.group.externalIdentity.externalId,
                            request.group.externalIdentity.typeId,
                        )
                    )
                },
                identities = identities,
            )
        }

        fun toReplaceGroupSpecification(request: ReplaceGroupRequest): ReplaceGroupSpecification {
            val identities = toIdentities(request.identitiesList)

            return ReplaceGroupSpecification(if (request.group.identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                IdentityIdOneOf(
                    UUID.fromString(request.group.id)
                )
            } else {
                IdentityIdOneOf(
                    IdentityExternalId(
                        request.group.externalIdentity.externalId,
                        request.group.externalIdentity.typeId,
                    )
                )
            }, identities)
        }
    }

}
