package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.interceptors;

import java.util.Objects;
import java.util.Optional;

import io.grpc.Status;
import io.grpc.StatusException;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import ru.yandex.intranet.imscore.util.MdcTaskDecorator;

/**
 * GRPC exception logging observer.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public class ExceptionLoggingResponseObserver<TResponse> implements StreamObserver<TResponse> {

    private static final Logger LOG = LoggerFactory.getLogger(ExceptionLoggingResponseObserver.class);

    private final StreamObserver<TResponse> delegate;
    private final String logId;

    public ExceptionLoggingResponseObserver(StreamObserver<TResponse> delegate, String logId) {
        this.delegate = delegate;
        this.logId = logId;
    }

    @Override
    public void onNext(TResponse value) {
        boolean cleanupPending = supplyMdc();
        try {
            delegate.onNext(value);
        } finally {
            if (cleanupPending) {
                MDC.remove(MdcTaskDecorator.LOG_ID_MDC_KEY);
            }
        }
    }

    @Override
    public void onError(Throwable t) {
        boolean cleanupPending = supplyMdc();
        try {
            delegate.onError(t);
            prepareException(t).ifPresent(e -> LOG.error("Unexpected error in GRPC service", e));
        } finally {
            if (cleanupPending) {
                MDC.remove(MdcTaskDecorator.LOG_ID_MDC_KEY);
            }
        }
    }

    @Override
    public void onCompleted() {
        delegate.onCompleted();
    }

    private Optional<Throwable> prepareException(Throwable throwable) {
        if (throwable instanceof StatusException || throwable instanceof StatusRuntimeException) {
            if (throwable.getCause() != null) {
                return Optional.of(throwable.getCause());
            } else if (isUnknown(throwable)) {
                return Optional.of(throwable);
            }
        }
        return Optional.empty();
    }

    private boolean isUnknown(Throwable throwable) {
        if (throwable instanceof StatusException) {
            Status.Code code = ((StatusException) throwable).getStatus().getCode();
            if (code == Status.Code.UNKNOWN || code == Status.Code.INTERNAL || code == Status.Code.DATA_LOSS) {
                return true;
            }
        }
        if (throwable instanceof StatusRuntimeException) {
            Status.Code code = ((StatusRuntimeException) throwable).getStatus().getCode();
            return code == Status.Code.UNKNOWN || code == Status.Code.INTERNAL || code == Status.Code.DATA_LOSS;
        }
        return false;
    }

    private boolean supplyMdc() {
        if (logId != null) {
            String mdcLogId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
            if (!Objects.equals(logId, mdcLogId)) {
                MDC.put(MdcTaskDecorator.LOG_ID_MDC_KEY, logId);
                return true;
            }
        } else {
            if (MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY) != null) {
                MDC.remove(MdcTaskDecorator.LOG_ID_MDC_KEY);
            }
        }
        return false;
    }

}
