package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.interceptors;

import io.grpc.Context;
import io.grpc.ServerCall;
import org.slf4j.MDC;

import ru.yandex.intranet.imscore.util.MdcTaskDecorator;

/**
 * Call listener to put log id into MDC.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public class LogIdCallListener<ReqT> extends ServerCall.Listener<ReqT> {

    private final ServerCall.Listener<ReqT> delegate;
    private final Context context;
    private final String logId;

    public LogIdCallListener(ServerCall.Listener<ReqT> delegate, Context context, String logId) {
        this.delegate = delegate;
        this.context = context;
        this.logId = logId;
    }

    @Override
    public void onMessage(ReqT message) {
        Context previous = context.attach();
        MDC.put(MdcTaskDecorator.LOG_ID_MDC_KEY, logId);
        try {
            delegate.onMessage(message);
        } finally {
            context.detach(previous);
            MDC.remove(MdcTaskDecorator.LOG_ID_MDC_KEY);
        }
    }

    @Override
    public void onHalfClose() {
        Context previous = context.attach();
        MDC.put(MdcTaskDecorator.LOG_ID_MDC_KEY, logId);
        try {
            delegate.onHalfClose();
        } finally {
            context.detach(previous);
            MDC.remove(MdcTaskDecorator.LOG_ID_MDC_KEY);
        }
    }

    @Override
    public void onCancel() {
        Context previous = context.attach();
        MDC.put(MdcTaskDecorator.LOG_ID_MDC_KEY, logId);
        try {
            delegate.onCancel();
        } finally {
            context.detach(previous);
            MDC.remove(MdcTaskDecorator.LOG_ID_MDC_KEY);
        }
    }

    @Override
    public void onComplete() {
        Context previous = context.attach();
        MDC.put(MdcTaskDecorator.LOG_ID_MDC_KEY, logId);
        try {
            delegate.onComplete();
        } finally {
            context.detach(previous);
            MDC.remove(MdcTaskDecorator.LOG_ID_MDC_KEY);
        }
    }

    @Override
    public void onReady() {
        Context previous = context.attach();
        MDC.put(MdcTaskDecorator.LOG_ID_MDC_KEY, logId);
        try {
            delegate.onReady();
        } finally {
            context.detach(previous);
            MDC.remove(MdcTaskDecorator.LOG_ID_MDC_KEY);
        }
    }

}
