package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.services

import com.google.protobuf.Any
import com.google.rpc.Status
import io.grpc.protobuf.StatusProto
import net.devh.boot.grpc.server.advice.GrpcAdvice
import net.devh.boot.grpc.server.advice.GrpcExceptionHandler
import org.slf4j.LoggerFactory
import ru.yandex.intranet.imscore.core.exceptions.CycleException
import ru.yandex.intranet.imscore.core.exceptions.ResourceAlreadyExistsException
import ru.yandex.intranet.imscore.core.exceptions.ResourceNotFoundException
import ru.yandex.intranet.imscore.core.exceptions.StillReferencedException
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.exceptions.ValidationException
import ru.yandex.intranet.imscore.proto.error.ErrorDetails
import ru.yandex.intranet.imscore.proto.error.FieldError

/**
 *  Grpc exception handler
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Suppress("unused")
@GrpcAdvice
class ExceptionHandler {
    private val log = LoggerFactory.getLogger(ExceptionHandler::class.java)

    @GrpcExceptionHandler(ValidationException::class)
    fun handleValidationException(ex: ValidationException): Exception {
        return StatusProto.toStatusRuntimeException(
            Status.newBuilder()
                .setCode(com.google.rpc.Code.INVALID_ARGUMENT.number)
                .setMessage(ex.message)
                .addDetails(Any.pack(ex.errorDetails))
                .build()
        )
    }

    @GrpcExceptionHandler(ResourceNotFoundException::class)
    fun handleException(ex: ResourceNotFoundException): Exception {
        val statusBuilder = Status.newBuilder()
            .setCode(com.google.rpc.Code.NOT_FOUND.number)
            .setMessage(ex.message)
        if (ex.causes != null && ex.causes.isNotEmpty()) {
            val errorDetails = ErrorDetails.newBuilder()
            ex.causes.forEachIndexed { _, it ->
                errorDetails.addFieldErrors(
                    FieldError.newBuilder()
                        .setCode(it.message)
                        .build()
                )
            }
            statusBuilder.addDetails(Any.pack(errorDetails.build()))
        }

        return StatusProto.toStatusRuntimeException(
            statusBuilder.build()
        )
    }

    @GrpcExceptionHandler(ResourceAlreadyExistsException::class)
    fun handleException(ex: ResourceAlreadyExistsException): Exception {
        return StatusProto.toStatusRuntimeException(
            Status.newBuilder()
                .setCode(com.google.rpc.Code.ALREADY_EXISTS.number)
                .setMessage(ex.message)
                .build()
        )
    }

    @GrpcExceptionHandler(IllegalArgumentException::class)
    fun handleException(ex: IllegalArgumentException): Exception {
        return StatusProto.toStatusRuntimeException(
            Status.newBuilder()
                .setCode(com.google.rpc.Code.INVALID_ARGUMENT_VALUE)
                .setMessage(ex.message)
                .build()
        )
    }

    @GrpcExceptionHandler(CycleException::class)
    fun handleException(ex: CycleException): Exception {
        return StatusProto.toStatusRuntimeException(
            Status.newBuilder()
                .setCode(com.google.rpc.Code.INVALID_ARGUMENT.number)
                .setMessage(ex.message)
                .build()
        )
    }

    @GrpcExceptionHandler(StillReferencedException::class)
    fun handleException(ex: StillReferencedException): Exception {
        return StatusProto.toStatusRuntimeException(
            Status.newBuilder()
                .setCode(com.google.rpc.Code.INVALID_ARGUMENT.number)
                .setMessage(ex.message)
                .build()
        )
    }

    @GrpcExceptionHandler(Exception::class)
    fun handleException(ex: Exception): Exception {
        log.error("Internal error while user request!", ex)
        return StatusProto.toStatusRuntimeException(
            Status.newBuilder()
                .setCode(com.google.rpc.Code.INTERNAL.number)
                .setMessage(ex.message)
                .build()
        )
    }

}
