package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.services.identity

import com.google.protobuf.Empty
import io.grpc.stub.StreamObserver
import net.devh.boot.grpc.server.service.GrpcService
import ru.yandex.intranet.imscore.core.services.identity.IdentityService
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.Grpc.toExceptionStreamObserver
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.converters.identity.IdentityProtoConverters
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.converters.identity.IdentityProtoConverters.Companion.toDeleteIdentitySpecifications
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.identity.IdentityProtoValidators
import ru.yandex.intranet.imscore.proto.identity.CreateIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.CreateIdentityResponse
import ru.yandex.intranet.imscore.proto.identity.DeleteIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.GetIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.GetIdentityResponse
import ru.yandex.intranet.imscore.proto.identity.IdentityServiceGrpc
import ru.yandex.intranet.imscore.proto.identity.ListIdentitiesRequest
import ru.yandex.intranet.imscore.proto.identity.ListIdentitiesResponse
import ru.yandex.intranet.imscore.proto.identity.MoveIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.UpdateIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.UpdateIdentityResponse

/**
 *  Identity grpc service implementation
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Suppress("unused")
@GrpcService
class GrpcIdentityImpl(
    private val identityService: IdentityService
): IdentityServiceGrpc.IdentityServiceImplBase() {
    override fun listIdentities(
        request: ListIdentitiesRequest,
        responseObserver: StreamObserver<ListIdentitiesResponse>
    ) {
        toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            val cursor = IdentityProtoValidators.validateListIdentitiesRequest(request)

            val spec = IdentityProtoConverters.toIdentityListSpecification(request, cursor)
            val identities = identityService.getListBySpec(spec)

            val response = IdentityProtoConverters.toListIdentitiesResponse(
                identities,
                spec.size,
                request.fieldMask
            )
            streamResponseObserver.onNext(response)
            streamResponseObserver.onCompleted()
        }
    }

    override fun getIdentity(
        request: GetIdentityRequest, responseObserver: StreamObserver<GetIdentityResponse>
    ) {
        toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            IdentityProtoValidators.validateGetIdentityByIdRequest(request)

            val spec = IdentityProtoConverters.toIdentitySpecification(request)
            val identityDomain = identityService.getById(spec)

            val response = IdentityProtoConverters.toGetIdentityResponse(identityDomain, request.fieldMask)
            streamResponseObserver.onNext(response)
            streamResponseObserver.onCompleted()
        }
    }

    override fun createIdentity(
        request: CreateIdentityRequest,
        responseObserver: StreamObserver<CreateIdentityResponse>
    ) {
        toExceptionStreamObserver(responseObserver) {
                streamResponseObserver ->
            IdentityProtoValidators.validateCreateIdentityRequest(request)

            val identity = identityService.create(IdentityProtoConverters.toCreateIdentitySpecification(request))
            val response = IdentityProtoConverters.toCreateIdentityResponse(identity)
            streamResponseObserver.onNext(response)
            streamResponseObserver.onCompleted()
        }
    }

    override fun updateIdentity(
        request: UpdateIdentityRequest,
        responseObserver: StreamObserver<UpdateIdentityResponse>
    ) {
        toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            IdentityProtoValidators.validateUpdateIdentityRequest(request)

            val identity = identityService.update(IdentityProtoConverters.toUpdateSpecification(request))
            val response = IdentityProtoConverters.toUpdateIdentityResponse(identity)
            streamResponseObserver.onNext(response)
            streamResponseObserver.onCompleted()
        }
    }

    override fun deleteIdentity(request: DeleteIdentityRequest, responseObserver: StreamObserver<Empty>) {
        toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            IdentityProtoValidators.validateDeleteIdentityRequest(request)
            val deleteIdentitySpecifications = toDeleteIdentitySpecifications(request)
            identityService.delete(deleteIdentitySpecifications)

            streamResponseObserver.onNext(Empty.getDefaultInstance())
            streamResponseObserver.onCompleted()
        }
    }

    override fun moveIdentity(
        request: MoveIdentityRequest,
        responseObserver: StreamObserver<Empty>
    ) {
        toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            IdentityProtoValidators.validateMoveIdentityRequest(request)

            identityService.move(IdentityProtoConverters.toMoveIdentitySpecification(request))
            streamResponseObserver.onNext(Empty.getDefaultInstance())
            streamResponseObserver.onCompleted()
        }
    }
}
