package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.services.identityGroup

import com.google.protobuf.Empty
import io.grpc.stub.StreamObserver
import net.devh.boot.grpc.server.service.GrpcService
import ru.yandex.intranet.imscore.core.services.identityGroup.IdentityGroupService
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.Grpc
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.converters.identityGroup.IdentityGroupProtoConverters
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.identityGroup.IdentityGroupProtoValidators
import ru.yandex.intranet.imscore.proto.identityGroup.AddToGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ExistsInGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ExistsInGroupResponse
import ru.yandex.intranet.imscore.proto.identityGroup.IdentityGroupServiceGrpc.IdentityGroupServiceImplBase
import ru.yandex.intranet.imscore.proto.identityGroup.ListIdentityGroupsRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ListIdentityGroupsResponse
import ru.yandex.intranet.imscore.proto.identityGroup.RemoveFromGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ReplaceGroupRequest

/**
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Suppress("unused")
@GrpcService
class GrpcIdentityGroupImpl(
    private val identityGroupService: IdentityGroupService
): IdentityGroupServiceImplBase() {

    override fun listIdentityGroups(
        request: ListIdentityGroupsRequest,
        responseObserver: StreamObserver<ListIdentityGroupsResponse>
    ) {
        Grpc.toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            val cursor = IdentityGroupProtoValidators.validateListIdentityGroupsRequest(request)

            val spec = IdentityGroupProtoConverters.toListIdentityGroupsSpecification(request, cursor)
            val identityRelations = identityGroupService.listIdentityGroups(spec)

            val response = IdentityGroupProtoConverters.toListIdentityGroupsResponse(
                identityRelations,
                spec.size,
            )
            streamResponseObserver.onNext(response)
            streamResponseObserver.onCompleted()
        }
    }

    override fun existsInGroup(
        request: ExistsInGroupRequest,
        responseObserver: StreamObserver<ExistsInGroupResponse>
    ) {
        Grpc.toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            IdentityGroupProtoValidators.validateExistsInGroupRequest(request)

            val spec = IdentityGroupProtoConverters.toIdentityRelationToGroupSpecification(request)
            val exists = identityGroupService.existsInGroup(spec)

            val response = IdentityGroupProtoConverters.toExistsInGroupResponse(
                exists
            )
            streamResponseObserver.onNext(response)
            streamResponseObserver.onCompleted()
        }
    }

    override fun addToGroup(
        request: AddToGroupRequest,
        responseObserver: StreamObserver<Empty>
    ) {
        Grpc.toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            IdentityGroupProtoValidators.validateAddToGroupRequest(request)

            identityGroupService.addToGroup(
                IdentityGroupProtoConverters.toAddIdentityRelationsToGroupSpecification(request),
            )

            streamResponseObserver.onNext(Empty.getDefaultInstance())
            streamResponseObserver.onCompleted()
        }
    }

    override fun removeFromGroup(
        request: RemoveFromGroupRequest,
        responseObserver: StreamObserver<Empty>
    ) {
        Grpc.toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            IdentityGroupProtoValidators.validateRemoveFromGroupRequest(request)

            identityGroupService.removeFromGroup(
                IdentityGroupProtoConverters.toRemoveIdentityRelationsToGroupSpecification(request),
            )

            streamResponseObserver.onNext(Empty.getDefaultInstance())
            streamResponseObserver.onCompleted()
        }
    }

    override fun replaceGroup(request: ReplaceGroupRequest, responseObserver: StreamObserver<Empty>) {
        Grpc.toExceptionStreamObserver(responseObserver) { streamResponseObserver ->
            IdentityGroupProtoValidators.validateReplaceGroup(request)

            val spec = IdentityGroupProtoConverters.toReplaceGroupSpecification(request)
            identityGroupService.replaceGroup(spec)

            streamResponseObserver.onNext(Empty.getDefaultInstance())
            streamResponseObserver.onCompleted()
        }
    }
}
