package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators


import com.google.gson.JsonParser
import com.google.protobuf.StringValue
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.PageTokenUtils.decode
import ru.yandex.intranet.imscore.proto.error.FieldError
import ru.yandex.intranet.imscore.proto.identity.ExternalIdentity
import ru.yandex.intranet.imscore.proto.identity.IdentityCompositeId
import ru.yandex.intranet.imscore.proto.identity.ModifiableIdentityData
import java.util.UUID
import java.util.function.BiConsumer
import java.util.function.Consumer
import java.util.function.Function

/**
 * Common grpc field validators
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */

class GrpcValidatorUtils {
    companion object {
        val defaultValue: String = StringValue.getDefaultInstance().value

        fun <T>validateRequired(field: String, value: T?, notDefaultString: Boolean = false): FieldError? {
            if (value == null) {
                return FieldError.newBuilder().setCode("Value is required").setKey(field).build()
            }
            val ok: Boolean = when (value) {
                is String -> {
                    value.isNotEmpty()
                        && (!notDefaultString || value != defaultValue)
                }
                else -> true
            }
            if (!ok) {
                return FieldError.newBuilder().setCode("Value is required").setKey(field).build()
            }
            return null
        }

        fun <T> validateOneOfIsSet(field: String, value: T?, notSetCase: T, fieldErrors: MutableList<FieldError>,
            required: Boolean): Boolean {
            if (value == null) {
                if (required) {
                    fieldErrors.add(
                        FieldError.newBuilder()
                            .setCode("OneOf is required")
                            .setKey(field)
                            .build()
                    )
                }
                return false
            }

            if (value == notSetCase) {
                if (required) {
                    fieldErrors.add(
                        FieldError.newBuilder()
                            .setCode("OneOf must be set")
                            .setKey(field)
                            .build()
                    )
                }
                return false
            }

            return true
        }

        fun validateListPageSize(pageSize: Int): MutableList<FieldError> {
            val fieldErrors = mutableListOf<FieldError>()

            var fieldError = validateIntLTE("page_size", pageSize, 1000)
            if (fieldError != null) {
                fieldErrors.add(fieldError)
            }

            fieldError = validateIntGTE("page_size", pageSize, 0)
            if (fieldError != null) {
                fieldErrors.add(fieldError)
            }
            return fieldErrors
        }

        fun validateUuidString(field: String, value: String?): FieldError? {
            try {
                UUID.fromString(value)
            } catch (e: IllegalArgumentException) {
                return FieldError.newBuilder().setCode("Value is not a UUID").setKey(field).build()
            }
            return null
        }

        fun validateIntLTE(field: String, value: Int, compare: Int): FieldError? {
            if (value > compare) {
                return FieldError.newBuilder()
                    .setCode(String.format("Value should be less than or equal to %d", compare))
                    .setKey(field).build()
            }
            return null
        }

        fun validateIntGTE(field: String, value: Int, compare: Int): FieldError? {
            if (value < compare) {
                return FieldError.newBuilder()
                    .setCode(String.format("Value should be greater than or equal to %d", compare))
                    .setKey(field).build()
            }
            return null
        }

        fun validateB64PageToken(encodedPageToken: String, expectedSize: Int?, func: Function<String, FieldError?>,
            baseField: String, fieldErrors: MutableList<FieldError>): List<String>? {
            if (encodedPageToken == defaultValue) {
                return null
            }

            val pageTokens = try {
                decode(encodedPageToken)
            } catch (e: Exception) {
                fieldErrors.add(FieldError.newBuilder().setCode("Value is not a page token").setKey(baseField).build())
                null
            }

            return if (fieldErrors.isEmpty()) {
                if (expectedSize != null && (pageTokens == null || pageTokens.size != expectedSize)) {
                    fieldErrors.add(FieldError.newBuilder().setCode("Value is not a page token").setKey(baseField).build())
                }

                if (fieldErrors.isEmpty() && pageTokens != null) {
                    for (pageToken in pageTokens) {
                        val fieldError = func.apply(pageToken)
                        if (fieldError != null) {
                            fieldErrors.add(FieldError.newBuilder().setCode("Value is not a page token")
                                .setKey(baseField).build())
                            return null
                        }
                    }
                }

                return pageTokens
            } else {
                pageTokens
            }
        }

        fun validateIdentityCompositeId(message: IdentityCompositeId, baseFiled: String,
            fieldErrors: MutableList<FieldError>, required: Boolean = true) {
            val identityIdOneofCase = message.identityIdOneofCase
            val validateOneOfIsSet = validateOneOfIsSet(
                baseFiled + "identity_id_oneof", identityIdOneofCase,
                IdentityCompositeId.IdentityIdOneofCase.IDENTITYIDONEOF_NOT_SET, fieldErrors, required
            )

            if (fieldErrors.isEmpty()) {
                if (!validateOneOfIsSet && !required) {
                    return
                }

                if (identityIdOneofCase == IdentityCompositeId.IdentityIdOneofCase.ID) {
                    val validateUuidString = validateUuidString(baseFiled + "identity_id_oneof.id",
                        message.id)
                    if (validateUuidString != null) {
                        fieldErrors.add(validateUuidString)
                    }
                } else {
                    validateExternalIdentity(message.externalIdentity,
                        baseFiled + "identity_id_oneof", fieldErrors)
                }
            }
        }

        fun validateExternalIdentity(message: ExternalIdentity, baseFiled: String?, fieldErrors: MutableList<FieldError>) {
            val field = if (baseFiled == null) {
                "external_identity"
            } else {
                "$baseFiled.external_identity"
            }
            val fieldError = validateRequired(
                "$field.external_id",
                message.externalId)
            if (fieldError != null) {
                fieldErrors.add(fieldError)
            }

            validateExternalIdentityTypeId(message, field, fieldErrors)
        }

        fun validateExternalIdentityTypeId(message: ExternalIdentity, baseFiled: String?,
            fieldErrors: MutableList<FieldError>) {
            val fieldError = validateRequired(
                "$baseFiled.type_id",
                message.typeId)
            if (fieldError != null) {
                fieldErrors.add(fieldError)
            }
        }

        fun validateRepeatedIdentityCompositeId(repeatedField: List<IdentityCompositeId>, baseField: String,
            fieldErrors: MutableList<FieldError>) {
            validateRepeatedField(repeatedField, baseField) { identityCompositeId, field ->
                validateIdentityCompositeId(identityCompositeId,field, fieldErrors)
            }
        }

        fun <T> validateRepeatedField(repeatedField: List<T>, baseField: String, consumer: BiConsumer<T, String>) {
            for (i in repeatedField.indices) {
                consumer.accept(repeatedField[i], "${baseField}.${i}.")
            }
        }

        fun <T> validateNullOrValidT(value: T?, consumer: Consumer<T>) {
            if (value != null) {
                consumer.accept(value)
            }
        }

        fun validateDefaultOrValidString(value: String, consumer: Consumer<String>) {
            if (value != defaultValue) {
                consumer.accept(value)
            }
        }

        fun validateValidJson(json: String?, field: String, fieldErrors: MutableList<FieldError>) {
            try {
                JsonParser.parseString(json).asJsonObject
            } catch (e: Exception) {
                fieldErrors.add(
                    FieldError.newBuilder().setCode("Value is invalid json").setKey(field).build()
                )
            }
        }

        fun validateModifiableIdentityData(identityData: ModifiableIdentityData?, baseField: String,
            fieldErrors: MutableList<FieldError>, isUpdate: Boolean) {
            if (identityData == null) {
                return
            }

            if (isUpdate) {
                validateNullOrValidT(
                    if (identityData.hasAdditionalData()) {
                        identityData.additionalData.value
                    } else {
                        null
                    }
                ) {
                        value -> validateDefaultOrValidString(value) {
                        json -> validateValidJson(json, "$baseField.additional_data", fieldErrors)
                } }
            } else {
                validateNullOrValidT(
                    if (identityData.hasAdditionalData()) {
                        identityData.additionalData.value
                    } else {
                        null
                    }
                ) { json -> validateValidJson(json, "$baseField.additional_data", fieldErrors) }
            }
        }
    }
}
