package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.identity

import com.google.rpc.Code
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.exceptions.ValidationException
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateB64PageToken
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateDefaultOrValidString
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateExternalIdentityTypeId
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateIdentityCompositeId
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateListPageSize
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateModifiableIdentityData
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateUuidString
import ru.yandex.intranet.imscore.proto.error.ErrorDetails
import ru.yandex.intranet.imscore.proto.error.FieldError
import ru.yandex.intranet.imscore.proto.identity.CreateIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.DeleteIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.GetIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.ListIdentitiesRequest
import ru.yandex.intranet.imscore.proto.identity.MoveIdentityRequest
import ru.yandex.intranet.imscore.proto.identity.UpdateIdentityRequest
import java.util.UUID

/**
 * Identity grpc validators
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
class IdentityProtoValidators {
    companion object {
        fun validateListIdentitiesRequest(req: ListIdentitiesRequest): UUID? {
            val fieldErrors = mutableListOf<FieldError>()

            val tokenList = validateB64PageToken(req.pageToken, 1,
                { pageToken -> validateUuidString("page_token", pageToken) }, "page_token", fieldErrors
            )

            fieldErrors.addAll(validateListPageSize(req.pageSize))
            validateDefaultOrValidString(req.groupId) { str -> validateUuidString("group_id", str) }

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    Code.INVALID_ARGUMENT.name,
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }

            return if (tokenList != null) {
                UUID.fromString(tokenList[0])
            } else {
                null
            }
        }

        fun validateGetIdentityByIdRequest(req: GetIdentityRequest) {
            val fieldErrors = arrayListOf<FieldError>()

            validateIdentityCompositeId(req.identity, "identity.", fieldErrors)

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    "identity.identity_id_oneof is invalid",
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }
        }

        fun validateCreateIdentityRequest(req: CreateIdentityRequest) {
            val fieldErrors = mutableListOf<FieldError>()
            validateExternalIdentityTypeId(req.externalIdentity, "external_identity", fieldErrors)
            validateIdentityCompositeId(req.parentId, "parent_id.", fieldErrors, false)
            validateModifiableIdentityData(req.data, "data", fieldErrors , false)

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    "CreateIdentityRequest is invalid",
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }
        }

        fun validateUpdateIdentityRequest(req: UpdateIdentityRequest) {
            val fieldErrors = mutableListOf<FieldError>()
            validateIdentityCompositeId(req.identity, "identity.", fieldErrors)
            validateModifiableIdentityData(req.data, "data", fieldErrors , true)

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    "UpdateIdentityRequest is invalid",
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }
        }

        fun validateDeleteIdentityRequest(request: DeleteIdentityRequest) {
            val fieldErrors = mutableListOf<FieldError>()
            validateIdentityCompositeId(request.identity, "identity.", fieldErrors)

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    "id_oneof is invalid",
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }
        }

        fun validateMoveIdentityRequest(request: MoveIdentityRequest) {
            val fieldErrors = mutableListOf<FieldError>()
            validateIdentityCompositeId(request.identity, "identity.", fieldErrors)
            validateIdentityCompositeId(request.toGroup, "group.", fieldErrors)

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    "identity_id_oneof is invalid",
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }
        }
    }
}
