package ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.identityGroup

import com.google.rpc.Code
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.exceptions.ValidationException
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateB64PageToken
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateIdentityCompositeId
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateListPageSize
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateRepeatedIdentityCompositeId
import ru.yandex.intranet.imscore.infrastructure.presentation.grpc.validators.GrpcValidatorUtils.Companion.validateUuidString
import ru.yandex.intranet.imscore.proto.error.ErrorDetails
import ru.yandex.intranet.imscore.proto.error.FieldError
import ru.yandex.intranet.imscore.proto.identity.IdentityCompositeId
import ru.yandex.intranet.imscore.proto.identityGroup.AddToGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ExistsInGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ListIdentityGroupsRequest
import ru.yandex.intranet.imscore.proto.identityGroup.RemoveFromGroupRequest
import ru.yandex.intranet.imscore.proto.identityGroup.ReplaceGroupRequest
import java.util.UUID

/**
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
class IdentityGroupProtoValidators {

    companion object {

        fun validateListIdentityGroupsRequest(req: ListIdentityGroupsRequest): UUID? {
            val fieldErrors = mutableListOf<FieldError>()

            validateIdentityCompositeId(req.identity, "identity.", fieldErrors)

            val tokenList = validateB64PageToken(req.pageToken, 1,
                { pageToken -> validateUuidString("page_token", pageToken) }, "page_token", fieldErrors
            )

            fieldErrors.addAll(validateListPageSize(req.pageSize))

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    Code.INVALID_ARGUMENT.name,
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }

            return if (tokenList != null) {
                UUID.fromString(tokenList[0])
            } else {
                null
            }
        }

        fun validateExistsInGroupRequest(req: ExistsInGroupRequest) {
            val fieldErrors = mutableListOf<FieldError>()

            validateIdentityCompositeId(req.group, "group.", fieldErrors)

            validateIdentityCompositeId(req.identity, "identity.", fieldErrors)

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    Code.INVALID_ARGUMENT.name,
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }
        }

        fun validateAddToGroupRequest(req: AddToGroupRequest) {
            val fieldErrors = mutableListOf<FieldError>()

            validateGroupsWithIdentityList(req.group, "group.", req.identitiesList, "identity", fieldErrors)

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    Code.INVALID_ARGUMENT.name,
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }
        }

        fun validateRemoveFromGroupRequest(req: RemoveFromGroupRequest) {
            val fieldErrors = mutableListOf<FieldError>()

            validateGroupsWithIdentityList(req.group, "group.", req.identitiesList, "identity", fieldErrors)

            if (fieldErrors.isNotEmpty()) {
                throw ValidationException(
                    Code.INVALID_ARGUMENT.name,
                    ErrorDetails.newBuilder().addAllFieldErrors(fieldErrors).build()
                )
            }
        }

        fun validateReplaceGroup(request: ReplaceGroupRequest) {
            val fieldErrors = mutableListOf<FieldError>()

            validateGroupsWithIdentityList(request.group, "group.",
                request.identitiesList, "identity", fieldErrors)
        }

        @Suppress("SameParameterValue")
        private fun validateGroupsWithIdentityList(group: IdentityCompositeId, groupField: String,
            identityList: List<IdentityCompositeId>, identityListField: String, fieldErrors: MutableList<FieldError>) {
            validateIdentityCompositeId(group, groupField, fieldErrors)
            validateRepeatedIdentityCompositeId(identityList, identityListField, fieldErrors)
        }
    }

}
