package ru.yandex.intranet.imscore.metrics;

import java.util.concurrent.ConcurrentHashMap;

import io.grpc.Status;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * GRPC server metrics.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class GrpcServerMetrics {

    private static final String RATE = "grpc.server.requests_rate";
    private static final String DURATION = "grpc.server.requests_duration_millis";
    private static final String CODE = "grpc_code";
    private static final String ALL = "all";

    private final Rate totalRate;
    private final Histogram totalDuration;
    private final ConcurrentHashMap<String, Rate> perCodeRate = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Histogram> perCodeDuration = new ConcurrentHashMap<>();

    public GrpcServerMetrics() {
        this.totalRate = MetricRegistry.root().rate(RATE, Labels.of(CODE, ALL));
        this.totalDuration = MetricRegistry.root().histogramRate(DURATION, Labels.of(CODE, ALL),
                Histograms.exponential(22, 2.0d, 1.0d));
        this.perCodeRate.put(Status.Code.UNKNOWN.name(),
                MetricRegistry.root().rate(RATE, Labels.of(CODE, Status.Code.UNKNOWN.name())));
    }

    public void onRequestCompletion(Status.Code code, long durationMillis) {
        totalRate.inc();
        totalDuration.record(durationMillis);
        String codeName = code.name();
        Rate codeRate = perCodeRate.computeIfAbsent(codeName,
                key -> MetricRegistry.root().rate(RATE, Labels.of(CODE, codeName)));
        codeRate.inc();
        Histogram codeDuration = perCodeDuration.computeIfAbsent(codeName,
                key -> MetricRegistry.root().histogramRate(DURATION, Labels.of(CODE, codeName),
                        Histograms.exponential(22, 2.0d, 1.0d)));
        codeDuration.record(durationMillis);
    }

}
