package ru.yandex.intranet.imscore.metrics;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Profile;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.imscore.metrics.consumer.PushMetricStatefulConsumer;
import ru.yandex.intranet.imscore.util.OneShotStopWatch;
import ru.yandex.monlib.metrics.CompositeMetricSupplier;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.encode.MetricFormat;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.nio.charset.StandardCharsets.UTF_8;
import static ru.yandex.monlib.metrics.encode.MetricEncoderFactory.createEncoder;

/**
 * Metrics delivery to solomon task.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
@Profile({"testing", "production"})
public class MetricsDeliveryToSolomonTask {

    private static final Logger LOG = LoggerFactory.getLogger(MetricsDeliveryToSolomonTask.class);
    private static final String SPACK_MIME_TYPE = "application/x-solomon-spack";
    private static final MetricFormat FORMAT = MetricFormat.SPACK;

    private final String solomonUrl;
    private final String solomonPushApi;
    private final Integer solomonPort;
    private final String solomonToken;
    private final String project;
    private final String service;
    private final String cluster;
    private final long intervalMillis;
    private final int connectTimeout;
    private final int readTimeout;
    private final String dc;
    private final String podId;
    private final Map<Labels, Long> rateMetricsState;

    @SuppressWarnings("ParameterNumber")
    public MetricsDeliveryToSolomonTask(
            @Value("${metrics.solomon_url}")
            String solomonUrl,
            @Value("${metrics.solomon_push_api}")
            String solomonPushApi,
            @Value("${metrics.solomon_port}")
            Integer solomonPort,
            @Value("${metrics.solomon_token}")
            String solomonToken,
            @Value("${metrics.project}")
            String project,
            @Value("${metrics.service}")
            String service,
            @Value("${metrics.cluster}")
            String cluster,
            @Value("${metrics.refreshDelayMs}")
            Integer refreshDelayMs,
            @Value("${metrics.connectTimeout}")
            int connectTimeout,
            @Value("${metrics.readTimeout}")
            int readTimeout,
            @Value("${deploy.dc}")
            String dc,
            @Value("${deploy.pod_id}")
            String podId
    ) {
        this.solomonUrl = solomonUrl;
        this.solomonPushApi = solomonPushApi;
        this.solomonPort = solomonPort;
        this.solomonToken = solomonToken;
        this.project = project;
        this.service = service;
        this.cluster = cluster;
        this.intervalMillis = refreshDelayMs;
        this.rateMetricsState = new HashMap<>();
        this.connectTimeout = connectTimeout;
        this.readTimeout = readTimeout;
        this.dc = dc;
        this.podId = podId;
    }

    @Scheduled(fixedDelayString = "${metrics.refreshDelayMs}",
            initialDelayString = "${metrics.refreshInitialDelayMs}")
    public void pushMetrics() {
        push();
    }

    private void push() {
        LOG.info("Starting push metrics to solomon");

        OneShotStopWatch oneShotStopWatch = new OneShotStopWatch();
        boolean success = false;
        String response = "";
        int status = -1;

        try {

            String query = String.format("project=%s&cluster=%s&service=%s", project, cluster, service);
            URL originalUrl = new URL(solomonUrl + solomonPushApi + "?" + query);
            URL url = new URL(originalUrl.getProtocol(), originalUrl.getHost(), solomonPort,
                    originalUrl.getFile());

            HttpURLConnection connection = null;

            try {
                connection = (HttpURLConnection) url.openConnection();
                connection.setDoOutput(true);
                connection.setRequestMethod("POST");
                connection.setConnectTimeout(connectTimeout);
                connection.setReadTimeout(readTimeout);
                connection.setRequestProperty("Content-Type", SPACK_MIME_TYPE);
                connection.setRequestProperty("Authorization", "OAuth " + solomonToken);

                try (OutputStream os = connection.getOutputStream()) {
                    encode(new CompositeMetricSupplier(List.of(MetricRegistry.root())), os);
                }

                connection.connect();

                status = connection.getResponseCode();
                try (BufferedReader br = 100 <= status && status <= 399
                        ? new BufferedReader(new InputStreamReader(connection.getInputStream(), UTF_8))
                        : new BufferedReader(new InputStreamReader(connection.getErrorStream(), UTF_8))) {
                    try (ByteArrayOutputStream buf = new ByteArrayOutputStream()) {
                        int result2 = br.read();
                        while (result2 != -1) {
                            buf.write((byte) result2);
                            result2 = br.read();
                        }
                        response = buf.toString(UTF_8);
                    }
                }

            } finally {
                if (connection != null) {
                    connection.disconnect();
                }
            }

            success = true;

        } catch (Exception ex) {
            LOG.error("Error while push metrics to solomon!", ex);
        } finally {
            long elapsed = oneShotStopWatch.elapsed(TimeUnit.MILLISECONDS);
            LOG.info(String.format("Ending push metrics to solomon; success = [%s]; time = [%s] ms; response = [%s]; " +
                    "status = [%s]", success, elapsed, response, status));
        }
    }

    private void encode(MetricSupplier supplier, OutputStream os) {
        // calculate grid timestamp
        long now = System.currentTimeMillis();
        long tsMillisByGrid = now - (now % intervalMillis);

        try (MetricEncoder encoder = createEncoder(os, FORMAT, CompressionAlg.ZSTD)) {
            var consumer = new PushMetricStatefulConsumer(encoder, rateMetricsState, (int) (intervalMillis / 1000));
            consumer.onStreamBegin(supplier.estimateCount());
            consumer.onCommonTime(tsMillisByGrid);
            supplier.append(0, Labels.of(
                    "DC", dc,
                    "POD_ID", podId
            ), consumer);
            consumer.onStreamEnd();
        } catch (Exception e) {
            throw new IllegalStateException("cannot encode sensors", e);
        }
    }

}
