import asyncio
import importlib
import logging

import click

from intranet.ims.staff_connector.staff_connector import sqs, settings

__package__ = 'intranet.ims.staff_connector.staff_connector'

log = logging.getLogger('staff_connector')


async def produce(sqs_client, handler, regular: int = 0):
    queue = await sqs.get_or_create_queue(
        handler.QUEUE,
        sqs_client=sqs_client,
        **settings.QUEUES_ATTRIBUTES[handler.QUEUE].dict(),
    )
    while True:
        await handler.produce_messages(queue)
        if not regular:
            break
        await asyncio.sleep(regular)


async def consume(sqs_client, handler, standalone: bool = False):
    queue = await sqs.get_or_create_queue(
        handler.QUEUE,
        sqs_client=sqs_client,
        **settings.QUEUES_ATTRIBUTES[handler.QUEUE].dict(),
    )
    while True:
        await sqs.consume_messages(queue, handler.process_message)
        if not standalone:
            break


async def entrypoint(coro_func, *args, **kwargs):
    async with sqs.get_sqs_context() as sqs_client:
        await coro_func(sqs_client, *args, **kwargs)


@click.group()
def cli():
    pass


@cli.command()
@click.argument('handler')
@click.option('--regular', type=int, default=0)
def producer(handler: str, regular: int):
    log.info('Start producing...')
    try:
        handler_module = importlib.import_module(f'{__package__}.handlers.{handler}')
    except ImportError:
        raise click.NoSuchOption(handler)

    try:
        asyncio.run(entrypoint(produce, handler_module, regular=regular))
    except KeyboardInterrupt:
        pass
    finally:
        log.info('Stop producing...')


@cli.command()
@click.argument('handler')
@click.option('--standalone', is_flag=True)
def consumer(handler: str, standalone: bool):
    log.info('Start consuming...')
    try:
        handler_module = importlib.import_module(f'{__package__}.handlers.{handler}')
    except ImportError:
        raise click.NoSuchOption(handler)
    try:
        asyncio.run(entrypoint(consume, handler_module, standalone=standalone))
    finally:
        log.info('Stop consuming...')


cli()
