import logging
import sys
from typing import Dict

import connection
import flow_logic


logging.basicConfig(stream=sys.stdout, level=logging.INFO)
log = logging.getLogger(__name__)


def run_script(
    code: str,
    context_params: Dict,
) -> None:
    code = code.replace('\r', '').strip()
    context = flow_logic.create_context(context_params)
    try:
        exec(code, context)
    except Exception as e:
        flow_logic.on_exception(context, e)
        log.info('Code execution interrupted by exception {}'.format(str(e.__class__)))
        return
    log.info('Code execution completed normally')
    flow_logic.on_complete(context)


def main():
    log.info('Started app')
    with connection.get() as connect:
        flow_logic.set_handlers(connect.send_data, connect.get_data)
        log.info('Sending starting signal')
        connect.send_data(0, {'status': 'container_start'})
        while True:
            log.info('Waiting for data')
            _, data = connect.get_data()
            msg_type = data['type']
            log.info('Received {} uid: {}, readable_id: {}'.format(
                msg_type,
                data.get('uid'),
                data.get('readable_id'),
            ))
            if msg_type == 'run_script':
                code, context = data['code'], data['context']
                run_script(code, context)
            elif msg_type == 'stop_container':
                connect.send_data(0, {'status': 'closing_app'})
                break


if __name__ == '__main__':
    main()
