import sform
import waffle

from uuid import uuid4

from django.core import validators
from django.core.exceptions import ValidationError

from ok.api.approvements.validators import validate_issue_key
from ok.api.core.fetchers import query_params_data_getter
from ok.api.core.fields import JsonField, QueueMultipleChoiceField
from ok.api.core.forms import UserValidationFormMixin
from ok.api.core.errors import AlreadyExists
from ok.approvements.choices import (
    APPROVEMENT_SORTING_TYPES,
    APPROVEMENT_STAGE_APPROVEMENT_SOURCES,
    APPROVEMENT_STATUSES,
    APPROVEMENT_TYPES,
    APPROVEMENT_STAGE_ACTIVE_STATUSES,
    POSSIBLE_DISAPPROVAL_REASONS,
)
from ok.approvements.models import Approvement, ApprovementStage
from ok.tracker.models import Queue
from ok.utils.strings import str_to_md5


class ApprovementFormBase(sform.SForm):
    object_id = sform.CharField(max_length=32, state=sform.REQUIRED)
    uid = sform.CharField(max_length=32, state=sform.REQUIRED)
    groups = sform.GridField(
        field_instance=sform.CharField(
            max_length=255,
            state=sform.REQUIRED,
        ),
    )

    def clean_uid(self, uid):
        return str_to_md5(uid)


class ApprovementChildStageForm(sform.SForm):

    approver = sform.CharField(max_length=32)


class ApprovementParentStageForm(sform.SForm):

    approver = sform.CharField(max_length=32)
    need_all = sform.BooleanField()
    is_with_deputies = sform.BooleanField()
    stages = sform.GridField(
        field_instance=sform.FieldsetField(ApprovementChildStageForm),
    )
    need_approvals = sform.IntegerField()

    def clean(self):
        cleaned_data = super().clean()

        need_all = cleaned_data.pop('need_all', None)
        is_with_deputies = cleaned_data.get('is_with_deputies')

        if is_with_deputies:
            if need_all or cleaned_data['need_approvals']:
                raise ValidationError(
                    message='Specify only 1 of fields: is_with_deputies, need_all, need_approvals',
                    code='redundant_params',
                )
            cleaned_data['need_approvals'] = 1
        elif need_all:
            cleaned_data['need_approvals'] = len(cleaned_data.get('stages', []))
        elif cleaned_data['need_approvals'] is None:
            cleaned_data['need_approvals'] = 1

        return cleaned_data


class ApprovementEditableParamsForm(UserValidationFormMixin, sform.SForm):
    text = sform.CharField()
    stages = sform.GridField(
        field_instance=sform.FieldsetField(ApprovementParentStageForm),
    )
    is_parallel = sform.BooleanField()
    is_reject_allowed = sform.BooleanField()
    callback_url = sform.CharField()
    tvm_id = sform.IntegerField()
    disapproval_reasons = sform.GridField(
        field_instance=sform.CharField(max_length=255),
    )
    info_mails_to = sform.GridField(
        field_instance=sform.EmailField(),
    )

    def clean(self):
        cleaned_data = super().clean()

        disapproval_reasons = cleaned_data.get('disapproval_reasons')
        if disapproval_reasons:
            validate_disapproval_reasons(disapproval_reasons)
            cleaned_data['disapproval_reasons'] = list(set(disapproval_reasons))

        info_mails_receivers = cleaned_data.get('info_mails_to', [])
        for receiver_email in info_mails_receivers:
            if not receiver_email.endswith('@yandex-team.ru'):
                raise ValidationError(
                    '',
                    code='Wrong email domain. Must be @yandex-team.ru',
                )
        return cleaned_data


class ApprovementEditForm(ApprovementEditableParamsForm):

    def clean(self):
        raw_cleaned_data = super().clean()

        cleaned_data = {
            key: value
            for key, value in raw_cleaned_data.items()
            if value not in validators.EMPTY_VALUES or key == 'stages'
        }

        if not cleaned_data:
            raise sform.ValidationError('No known field chosen')

        return cleaned_data


class ApprovementCreateForm(ApprovementFormBase, ApprovementEditableParamsForm):

    uid = sform.CharField(max_length=32)
    type = sform.ChoiceField(choices=APPROVEMENT_TYPES, default=APPROVEMENT_TYPES.tracker)
    text = sform.CharField(state=sform.REQUIRED)
    users = sform.GridField(
        field_instance=sform.CharField(
            max_length=32,
            state=sform.REQUIRED,
        ),
    )
    is_auto_approving = sform.BooleanField(default=False)
    is_reject_allowed = sform.BooleanField(default=True)
    flow_name = sform.CharField()
    flow_context = JsonField()
    create_comment = sform.BooleanField(default=False)

    # Автора могут задавать только пользователи с определёнными правами,
    # сейчас эти права регулируются waffle-флагом
    author = sform.CharField()

    USER_FIELDS = (
        'users',
        'author',
    )

    def clean_uid(self, uid):
        if not uid:
            uid = uuid4().hex
        return super().clean_uid(uid)

    def _collect_logins(self):
        logins = super()._collect_logins()
        logins['stages'] = []

        for stage in self.cleaned_data.get('stages', []):
            if stage['approver']:
                logins['stages'].append(stage['approver'])
            for child in stage.get('stages', []):
                logins['stages'].append(child['approver'])
        return logins

    def get_field_state(self, name):
        if name == 'stages' and not self.data.get('users') and not self.data.get('flow_name'):
            return sform.REQUIRED
        return super().get_field_state(name)

    @classmethod
    def get_duplicate_info(cls, object_id, only_active=False):
        qs = Approvement.objects.filter(object_id=object_id)
        if only_active:
            qs = qs.exclude(status=APPROVEMENT_STATUSES.closed)
        duplicate_info = qs.values('uuid', 'uid').first()
        return duplicate_info

    def clean_object_id(self, object_id):
        duplicate_info = self.get_duplicate_info(object_id, only_active=True)
        if duplicate_info:
            raise AlreadyExists(params=duplicate_info)
        return object_id

    def validate_tracker_fields(self):
        object_id = self.cleaned_data.get('object_id')
        if object_id:
            validate_issue_key(object_id)

    def clean(self):
        cleaned_data = super().clean()
        is_tracker_approvement = cleaned_data['type'] == APPROVEMENT_TYPES.tracker
        if is_tracker_approvement:
            self.validate_tracker_fields()
        self.validate_user_fields(check_tracker=is_tracker_approvement)

        # Для обратной совместимости со старым фронтом переносим данные из users в stages
        # После OK-266 этот код можно будет удалить (+ удалить users из формы)
        if cleaned_data.get('users') and not cleaned_data.get('stages'):
            cleaned_data['stages'] = [{'approver': u} for u in cleaned_data['users']]
        cleaned_data.pop('users', None)

        return cleaned_data


class ApprovementListFilterForm(sform.SForm):

    default_getter = query_params_data_getter

    stage_statuses = sform.MultipleChoiceField(
        choices=APPROVEMENT_STAGE_ACTIVE_STATUSES,
        default=[APPROVEMENT_STAGE_ACTIVE_STATUSES.current],
        state=sform.REQUIRED,
    )
    queues = QueueMultipleChoiceField()

    sort = sform.ChoiceField(
        choices=APPROVEMENT_SORTING_TYPES,
        default=APPROVEMENT_SORTING_TYPES.modified_asc,
        state=sform.REQUIRED,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        user = self.base_initial.get('user')
        assert user is not None
        approvement_queues = (
            Approvement.objects
            .filter(stages__approver=user.login)
            .values('tracker_queue')
        )
        self.fields['queues'].queryset = (
            Queue.objects
            .filter(id__in=approvement_queues)
            .order_by('name')
        )

        if not waffle.switch_is_active('enable_stage_status_current'):
            stage_statuses = self.fields['stage_statuses']
            stage_statuses.default = [s for s, _ in APPROVEMENT_STAGE_ACTIVE_STATUSES]


class ApprovementCheckForm(ApprovementFormBase):

    author = sform.CharField()

    def get_field_state(self, name):
        if name == 'author' and not self.data.get('groups'):
            return sform.REQUIRED
        return super().get_field_state(name)


class ApprovementApproveForm(sform.SForm):

    # Авторизованный пользователь может не совпадать с согласующим данной стадии,
    # если он согласует не за себя
    approver = sform.CharField()

    stages = sform.MultipleSuggestField(
        queryset=ApprovementStage.objects.none(),
        label_fields=['approver'],
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        assert 'active_stages' in self.initial
        self.fields['stages'].queryset = self.initial['active_stages']

    def get_field_state(self, name):
        if name == 'approver' and not self.data.get('stages'):
            return sform.REQUIRED
        return super().get_field_state(name)

    def clean(self):
        super().clean()
        self.cleaned_data['approvement_source'] = APPROVEMENT_STAGE_APPROVEMENT_SOURCES.api
        return self.cleaned_data


class ApprovementRejectForm(sform.SForm):

    disapproval_reason = sform.CharField()

    def clean(self):
        super().clean()
        disapproval_reason = self.cleaned_data.get('disapproval_reason')
        if disapproval_reason:
            validate_disapproval_reasons([disapproval_reason])
        return self.cleaned_data


def validate_disapproval_reasons(disapproval_reasons):
    invalid_reasons = set(disapproval_reasons) - {slug for slug, name in POSSIBLE_DISAPPROVAL_REASONS}
    if invalid_reasons:
        raise ValidationError(
            '',
            code=f'Invalid disapproval reasons: {", ".join(str(reason) for reason in  invalid_reasons)}',
        )
