import logging
import waffle

from django.db import IntegrityError
from django.db.models import prefetch_related_objects
from django.http.response import Http404
from django.utils.decorators import method_decorator
from django.views.decorators.csrf import ensure_csrf_cookie
from ids.exceptions import BackendError
from model_utils import Choices
from rest_framework.response import Response
from ylog.context import log_context

from ok.api.core.views import (
    BaseView,
    BaseViewSet,
    WorkflowView,
    unfold_query_params,
)
from ok.api.core.errors import Http400, AlreadyExists
from ok.approvements import choices
from ok.approvements.controllers import (
    ApprovementController,
    NoStagesForFlowException,
    ExceededApproversLimit,
    get_staff_group_member_logins,
    fill_staff_users,
)
from ok.approvements.helpers import prefetch_stage_parents
from ok.approvements.models import Approvement, ApprovementStage
from ok.approvements.workflow import ApprovementWorkflow
from ok.flows.executor import FlowNotFound
from ok.tracker.queues import get_queue_by_object_id
from ok.utils.staff import get_gap_newhire_data, GapNewhireCountsApiError

from . import serializers, forms


logger = logging.getLogger(__name__)


_approvement_prefetch = (
    'stages__history',
    'tracker_queue',
)


@method_decorator(ensure_csrf_cookie, 'dispatch')
class ApprovementView(BaseViewSet):

    model_class = Approvement
    detail_serializer_class = serializers.ApprovementSerializer
    list_item_serializer_class = serializers.ApprovementListItemSerializer
    validator_class = forms.ApprovementCreateForm

    def get_queryset(self):
        return (
            super().get_queryset()
            .prefetch_related(*_approvement_prefetch)
        )

    def filter_queryset(self, queryset):
        params = unfold_query_params(
            query_params=self.request.query_params,
            list_fields=[
                'queues',
                'stage_statuses',
            ],
        )
        filter_form = forms.ApprovementListFilterForm(
            data=params,
            base_initial={'user': self.request.yauser},
        )
        self.validate(filter_form)
        filter_params = filter_form.cleaned_data

        approver_stages = (
            ApprovementStage.objects
            .filter(
                approver=self.request.yauser.login,
                status__in=filter_params['stage_statuses'],
            )
            .order_by()
        )

        # Note: пока в этом списке работаем только с согласованиями, где пользователь согласующий
        queryset = queryset.filter(id__in=approver_stages.values('approvement_id'))

        if filter_params['queues']:
            queryset = queryset.filter(tracker_queue__in=filter_params['queues'])

        queryset = queryset.order_by(filter_params['sort'])
        return queryset

    def perform_create(self, data):
        with log_context(approvement_uid=data['uid']):
            # Явно задавать автора согласований могут только пользователи,
            # для которых включен флаг
            is_author_defined = (
                waffle.flag_is_active(self.request, 'can_set_approvement_author')
                and data['author']
            )
            initiator = self.request.yauser.login
            if not is_author_defined:
                data['author'] = initiator
            try:
                approvement = ApprovementController.create(data, initiator)
            except NoStagesForFlowException:
                raise Http400(code='No stages for flow')
            except FlowNotFound as exc:
                raise Http404(f'Flow {exc.flow_name} not found')
            except ExceededApproversLimit:
                raise Http400(code='Exceeded approvers limit')
            except IntegrityError as e:
                # Чаще всего IntegrityError случается, если одновременно пришли два одинаковых
                # запроса на создание согласования (из-за наружения уникальности object_id, uid)
                duplicate_info = forms.ApprovementCreateForm.get_duplicate_info(data['object_id'])
                if duplicate_info:
                    raise AlreadyExists(params=duplicate_info)
                else:
                    raise e from None

            approvement = ApprovementController(approvement).perform_auto_approve()

            prefetch_related_objects([approvement], *_approvement_prefetch)
            return approvement

    def get_initial_data(self):
        initial_data = unfold_query_params(
            query_params=self.request.query_params,
            list_fields=('users', 'groups'),
        )

        # Note: Мы не можем просто избавиться от поддержки поля users в предустановленных
        # параметрах согласования, потому что это используется в Фемиде и, вероятно,
        # в каких-нибудь макросах или триггерах.
        users = initial_data.get('users', [])
        self.parse_context_fields(initial_data)
        if 'flow_name' in initial_data:
            flow_name = initial_data['flow_name']
            try:
                result = ApprovementController.get_data_from_flow(
                    flow_name=flow_name,
                    flow_context=initial_data.get('flow_context'),
                )
                stages = result['data'].pop('stages', None)
                initial_data.update(result['data'])
            except Exception as exc:
                logger.warning(f'Err while executing flow {flow_name}: {exc}')
                stages = []
        else:
            stages = [{'approver': u} for u in users]
        try:
            stages = fill_staff_users(stages)
        except Exception:
            pass
        initial_data['stages'] = stages
        initial_data['users'] = users
        return initial_data

    @staticmethod
    def parse_context_fields(initial_data):
        prefix = 'flow_context.'
        context = {}
        for key, value in initial_data.items():
            if key.startswith(prefix):
                field = key[len(prefix):]
                context[field] = value
        if context:
            initial_data.setdefault('flow_context', {}).update(context)


class GapNewhireCountsView(BaseView):

    def get(self, request, *args, **kwargs):
        try:
            gap_newhire_data = get_gap_newhire_data(request)
        except GapNewhireCountsApiError:
            gap_newhire_data = {}
        return Response({
                'gap_count': gap_newhire_data.get('gap'),
                'newhire_count': gap_newhire_data.get('newhire'),
            })


@method_decorator(ensure_csrf_cookie, 'dispatch')
class ApprovementListFilterFormView(BaseViewSet):

    model_class = Approvement
    validator_class = forms.ApprovementListFilterForm

    def get_initial_data(self):
        return unfold_query_params(
            query_params=self.request.query_params,
            list_fields=[
                'queues',
                'stage_statuses',
            ],
        )


@method_decorator(ensure_csrf_cookie, 'dispatch')
class ApprovementCheckView(BaseView):
    """
    Класс проверки возможности создать или посмотреть согласование

    Роль ручки check заключается в том, что изначально фронт из iframe обращается к ней.
    По ее ответу определяется, что отображать:
     * форму создания (дергается create-form),
     * активное согласование (дергается detail) или
     * сообщение о ошибке (бек не дергается).
    """
    STATES = Choices(
        'exists',
        'can_create',
        'error_approvement_creating',
        'error_active_approvement_exists'
    )

    def get(self, request, *args, **kwargs):
        data = unfold_query_params(
            query_params=self.request.query_params,
            list_fields=('groups',),
        )
        check_form = forms.ApprovementCheckForm(data)
        self.validate(check_form)

        approvement, state = self.check_approvement(**check_form.cleaned_data)

        if approvement:
            tracker_queue = approvement.tracker_queue
        else:
            tracker_queue = get_queue_by_object_id(check_form.cleaned_data['object_id'])

        serializer = serializers.ApprovementCheckSerializer({
            'id': approvement.id if approvement else None,
            'uuid': approvement.uuid if approvement else None,
            'state': state,
            'tracker_queue': tracker_queue,
        })
        return Response(serializer.data)

    def check_approvement(self, object_id, uid, author, groups):
        object_approvements = Approvement.objects.filter(object_id=object_id)
        approvement = None
        state = None

        if object_approvements.exists():
            try:
                approvement = object_approvements.get(uid=uid)
                state = self.STATES.exists
            except Approvement.DoesNotExist:
                active_approvements = (
                    object_approvements
                    .exclude(status=choices.APPROVEMENT_STATUSES.closed)
                )
                # На данном тикете уже есть активное согласование
                if active_approvements.exists():
                    state = self.STATES.error_active_approvement_exists
        if state is None:
            state = self.get_new_approvement_state(author, groups)

        return approvement, state

    def get_new_approvement_state(self, author, groups):
        responsibles = {author} if author else set()
        try:
            responsibles |= get_staff_group_member_logins(groups) if groups else set()
        except BackendError:
            pass
        if self.request.yauser.login in responsibles:
            return self.STATES.can_create
        # Согласование уже настраивается другим пользователем
        return self.STATES.error_approvement_creating


class ApprovementWorkflowView(WorkflowView):

    model_class = Approvement
    detail_serializer_class = serializers.ApprovementSerializer
    workflow_class = ApprovementWorkflow

    def get_detail_serializer_object(self, instance, **kwargs):
        prefetch_related_objects([instance], *_approvement_prefetch)
        return super().get_detail_serializer_object(instance, **kwargs)


class ApprovementApproveView(ApprovementWorkflowView):

    action_name = 'approve'
    validator_class = forms.ApprovementApproveForm
    form_serializer_class = serializers.ApprovementApproveFormSerializer

    def get_validator_object(self, *args, **kwargs):
        initial = kwargs.setdefault('initial', {})
        initial['active_stages'] = (
            self.get_object().stages
            .active()
            .leaves()
            .prefetch_related(
                prefetch_stage_parents(),
            )
        )
        return super().get_validator_object(*args, **kwargs)


class ApprovementRejectView(ApprovementWorkflowView):

    action_name = 'reject'
    validator_class = forms.ApprovementRejectForm
    form_serializer_class = serializers.ApprovementRejectFormSerializer


@method_decorator(ensure_csrf_cookie, 'dispatch')
class ApprovementEditView(ApprovementWorkflowView):

    detail_serializer_class = serializers.ApprovementSerializer
    validator_class = forms.ApprovementEditForm
    action_name = 'edit'


@method_decorator(ensure_csrf_cookie, 'dispatch')
class ApprovementDisapprovalReasonsView(BaseView):
    def get(self, request, *args, **kwargs):
        return Response({slug: name for slug, name in choices.POSSIBLE_DISAPPROVAL_REASONS})
