import sform

from collections import OrderedDict

from django.core.exceptions import ValidationError


def _get_id_attr(f):
    return (
        getattr(f, 'to_field_name', None)
        or f.kwargs.get('to_field_name', None)
        or 'pk'
    )


# Пост-сериализаторы провалидированных данных для полей "модельного" типа.
# Сам сериализатор принимает 2 аргумента: (v)alue, (i)d_attr
_post_serializers = OrderedDict((
    (sform.MultipleSuggestField, lambda v, i: list(v.values_list(i, flat=True))),
    (sform.SuggestField, lambda v, i: getattr(v, i) if v else v),
    (sform.ModelMultipleChoiceField, lambda v, i: list(v.values_list(i, flat=True))),
    (sform.ModelChoiceField, lambda v, i: getattr(v, i) if v else v),
))


# Аргументы по умолчанию для всех вызовов метода clean у поля
_clean_kwargs = {
    # мы не хотим здесь валидировать "обязательность"
    'required': False,
    # этого ничего не нужно в clean, но аргументы обязательные
    'old_value': None,
    'trim_on_empty': None,
    'base_initial': None,
    'base_data': None,
}


def query_params_data_getter(form, field_name):
    """
    Получатель начальных данных для sform на основе сырых GET-параметров.
    Валидирует данные отдельно для каждого поля.
    Если валидация не успешна, не ругается, а отдает для поля значение по умолчанию.

    1. Не поддерживает FieldsetField, GridField и любые другие сложные вложенные поля.
    2. Не валидирует поле относительно других полей,
       потому что каждое поле валидируется по отдельности.

    :param form: sform.SForm - форма
    :param field_name: str - название поля
    :return: начальные данные для поля с названием `field_name`
    """
    field = form.fields[field_name]

    form.__dict__.setdefault('_cached_initial', {})
    form.__dict__.setdefault('_cached_serialized_initial', {})

    def _serialize(cleaned_value):
        for field_type, serializer in _post_serializers.items():
            if isinstance(field, field_type):
                return serializer(cleaned_value, _get_id_attr(field))
        return cleaned_value

    def fetch(initial):
        if field_name in form._cached_serialized_initial:
            return form._cached_serialized_initial[field_name]
        try:
            initial_value = initial[field_name]
            cleaned_value = field.clean(initial_value, **_clean_kwargs)
            if hasattr(form, 'clean_%s' % field_name):
                cleaned_value = getattr(form, 'clean_%s' % field_name)(cleaned_value)
            # Кэшируем сложные вычисленные объекты
            form._cached_initial[field_name] = cleaned_value
        except ValidationError:
            result = field.default
        else:
            result = _serialize(cleaned_value)

        form._cached_serialized_initial[field_name] = result
        return result

    return fetch
