import json

import sform

from django.core.exceptions import ValidationError

from ok.tracker.models import Queue


class JsonField(sform.Field):

    def clean(self, new_value, old_value, required, trim_on_empty, base_initial, base_data):
        if isinstance(new_value, (dict, list)):
            return new_value
        elif not new_value:
            if required:
                raise ValidationError(
                    'Required',
                    code='required',
                )
            return None
        elif isinstance(new_value, str):
            try:
                return json.loads(new_value)
            except json.JSONDecodeError:
                raise ValidationError(
                    'Value should be a serializable json string',
                    code='should_be_json',
                )


class QueueChoiceField(sform.ModelChoiceField):

    type_name = 'choice'

    def __init__(self, **kwargs):
        kwargs.setdefault('to_field_name', 'name')
        kwargs.setdefault('label_extractor', lambda x: x.name)
        kwargs.setdefault('queryset', Queue.objects.all())
        super().__init__(**kwargs)

    def clean(self, new_value, *args, **kwargs):
        new_value = self.prepare_value(new_value)
        return super().clean(new_value, *args, **kwargs)

    def prepare_value(self, value):
        if isinstance(value, str):
            value = value.upper()
        return value


class QueueMultipleChoiceField(QueueChoiceField, sform.ModelMultipleChoiceField):

    type_name = 'multiplechoice'

    def prepare_value(self, value):
        if isinstance(value, (list, tuple)):
            value = [super(QueueMultipleChoiceField, self).prepare_value(v) for v in value]
        return value

