from itertools import chain

from django.core.exceptions import ValidationError
from ids.exceptions import BackendError

from ok.tracker.queues import get_or_create_queue_by_object_id
from ok.utils.attrs import get_attribute
from ok.utils.staff import get_staff_users_iter


EMPTY_LABEL = '\u2014'
PG_MAX_INTEGER_VALUE = 2147483647
STAFF_USER_EXTERNAL = 'external'


class InvalidLoginsError(ValidationError):

    def __init__(self, invalid_logins=None):
        key = 'invalid_logins'
        params = {key: list(invalid_logins or [])}
        super(InvalidLoginsError, self).__init__(key, key, params)


class UserPermissionsError(ValidationError):

    def __init__(self, external_users=None):
        key = 'external_users'
        params = {key: external_users}
        super(UserPermissionsError, self).__init__(key, key, params)


class UserValidationFormMixin(object):
    """
    Миксин для формы, который позволяет валидировать все поля,
    которые принимают пользователей, за 1 поход в staff-api.
    Нужно в подклассе в `USER_FIELDS` перечислить поля, которые ожидают
    пользователей (логины), и в методе `clean` вызвать `validate_user_fields`
    """
    USER_FIELDS = ()

    def _collect_logins(self):
        result = {}
        for field in self.USER_FIELDS:
            logins = self.cleaned_data.get(field)
            if logins:
                logins = [logins] if isinstance(logins, str) else logins
                result[field] = logins
        return result

    def _get_valid_users(self, logins):
        try:
            users = list(get_staff_users_iter({
                'login': ','.join(logins),
                '_fields': 'login,official.affiliation',
            }))
            return {i['login']: get_attribute(i, 'official.affiliation', '') for i in users}
        except BackendError:
            raise ValidationError(
                message='staff_api_response_error',
                code='staff_api_response_error',
            )

    def validate_user_fields(self, check_tracker=True):
        field_to_logins = self._collect_logins()
        raw_logins = set(chain.from_iterable(field_to_logins.values()))
        logins_to_affiliation = self._get_valid_users(raw_logins)
        valid_logins = list(logins_to_affiliation.keys()) + ['']

        if check_tracker:
            queue = get_or_create_queue_by_object_id(self.data.get('object_id'))
            allow_externals = queue.allow_externals if queue else True
        else:
            allow_externals = True

        errors = {}
        for field, logins in field_to_logins.items():
            invalid_logins = set(logins) - set(valid_logins)
            if invalid_logins:
                errors[(field,)] = [InvalidLoginsError(invalid_logins)]
                continue

            if not allow_externals:
                external_users = []
                for login in logins:
                    if login == '':
                        continue

                    if logins_to_affiliation[login] == STAFF_USER_EXTERNAL:
                        external_users.append(login)

                if external_users:
                    errors.setdefault((field,), []).append(UserPermissionsError(external_users))

        if errors:
            raise ValidationError(errors)
