from django.core.exceptions import PermissionDenied
from django.shortcuts import get_object_or_404
from django.utils.decorators import classonlymethod
from django.utils.functional import cached_property
from rest_framework import status
from rest_framework.response import Response
from rest_framework.settings import api_settings
from rest_framework.views import APIView
from rest_framework.viewsets import ViewSetMixin
from django_tools_log_context import request_log_context

from ok.api.core.errors import ResponseError


def get_base_view_context(view):
    return {
        'user': view.request.yauser,
    }


class LogContextMixin:
    def dispatch(self, request, *args, **kwargs):
        with request_log_context(request, endpoint=self, threshold=0):
            return super(LogContextMixin, self).dispatch(request, *args, **kwargs)


class ValidationMixin:

    validator_class = None

    def get_validator_class(self):
        return self.validator_class

    def get_validator_object(self, *args, **kwargs):
        base_initial = kwargs.get('base_initial') or {}
        kwargs['base_initial'] = dict(get_base_view_context(self), **base_initial)
        validator_class = self.get_validator_class()
        return validator_class(*args, **kwargs) if validator_class else None

    def validate(self, validator):
        if not validator.is_valid():
            raise ResponseError(
                data=validator.errors,
                status=status.HTTP_400_BAD_REQUEST,
            )


class SerializationMixin:

    list_item_serializer_class = None
    detail_serializer_class = None
    form_serializer_class = None

    def get_detail_serializer_class(self):
        return self.detail_serializer_class

    def get_detail_serializer_object(self, instance, **kwargs):
        kwargs['context'] = dict(
            self.get_detail_serializer_context(),
            **get_base_view_context(self)
        )
        serializer_class = self.get_detail_serializer_class()
        self.setup_eager_loading(serializer_class, [instance])
        return serializer_class(instance, **kwargs) if serializer_class else None

    def get_detail_serializer_context(self):
        return {}

    def get_list_item_serializer_class(self):
        return self.list_item_serializer_class

    def get_list_item_serializer_object(self, collection, **kwargs):
        kwargs['context'] = dict(
            self.get_list_item_serializer_context(),
            **get_base_view_context(self)
        )
        serializer_class = self.get_list_item_serializer_class()
        return serializer_class(collection, many=True, **kwargs)

    def get_list_item_serializer_context(self):
        return {}

    def get_form_serializer_class(self):
        return self.form_serializer_class

    def get_form_serializer_object(self, instance, **kwargs):
        kwargs['context'] = dict(
            self.get_form_serializer_context(),
            **get_base_view_context(self)
        )
        serializer_class = self.get_form_serializer_class()
        return serializer_class(instance, **kwargs) if serializer_class else None

    def get_form_serializer_context(self):
        return {}


class BaseView(LogContextMixin, ValidationMixin, SerializationMixin, APIView):

    model_class = None
    pagination_class = api_settings.DEFAULT_PAGINATION_CLASS

    def filter_queryset(self, queryset):
        return queryset

    def get_queryset(self):
        return self.model_class.objects.all()

    def setup_eager_loading(self, serializer_class, collection):
        setup_eager_loading = getattr(serializer_class, 'setup_eager_loading', None)
        if setup_eager_loading is not None:
            collection = setup_eager_loading(collection)
        return collection

    def get_object(self):
        if not hasattr(self, '_instance'):
            id_field = 'uuid' if 'uuid' in self.kwargs else 'pk'
            queryset = self.get_queryset()
            self._instance = get_object_or_404(queryset, **{id_field: self.kwargs[id_field]})
            self.check_object_permissions(self.request, self._instance)
        return self._instance

    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        serializer_class = self.get_list_item_serializer_class()
        queryset = self.setup_eager_loading(serializer_class, queryset)
        paginator = self.pagination_class()
        self.page = paginator.paginate_queryset(queryset, request, view=self)
        serializer = self.get_list_item_serializer_object(self.page)
        return paginator.get_paginated_response(serializer.data)

    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_detail_serializer_object(instance)
        return Response(serializer.data)

    def create(self, request, *args, **kwargs):
        validator = self.get_validator_object(request.data)
        self.validate(validator)
        instance = self.perform_create(data=validator.cleaned_data)
        serializer = self.get_detail_serializer_object(instance)
        data = serializer.data if serializer else {}
        return Response(data, status=status.HTTP_201_CREATED)

    def perform_create(self, data):
        raise NotImplementedError

    def update(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_detail_serializer_object(instance)
        initial_data = serializer.data if serializer else {}
        validator = self.get_validator_object(request.data, initial=initial_data)
        self.validate(validator)
        instance = self.perform_update(
            data=validator.cleaned_data,
            instance=instance,
        )
        serializer = self.get_detail_serializer_object(instance)
        data = serializer.data if serializer else {}
        return Response(data, status=status.HTTP_200_OK)

    def perform_update(self, data, instance):
        raise NotImplementedError

    def destroy(self, request, *args, **kwargs):
        instance = self.get_object()
        self.perform_destroy(instance)
        return Response(status=status.HTTP_204_NO_CONTENT)

    def perform_destroy(self, instance):
        raise NotImplementedError()

    def create_form(self, request, *args, **kwargs):
        initial_data = self.get_initial_data()
        form = self.get_validator_object(initial=initial_data)
        return Response(form.as_dict())

    def update_form(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_form_serializer_object(instance)
        initial_data = serializer.data if serializer else {}
        form = self.get_validator_object(initial=initial_data)
        return Response(form.as_dict())

    @property
    def session_id(self):
        return self.request.COOKIES.get('Session_id')

    def get_initial_data(self):
        return {}


class BaseViewSet(ViewSetMixin, BaseView):
    pass


class WorkflowViewMixin:

    workflow_class = None
    action_name = None

    @cached_property
    def workflow(self):
        return self.workflow_class(
            instance=self.get_object(),
            user=self.request.yauser.login,
        )

    def perform_action(self, action_name):
        action = self.workflow.get_action(action_name)
        if not action.is_available():
            raise PermissionDenied

        validator = self.get_validator_object(self.request.data)
        if validator is None:
            data = {}
        else:
            self.validate(validator)
            data = validator.cleaned_data

        instance = action.perform(**data)
        serializer = self.get_detail_serializer_object(instance)
        response_data = serializer.data if serializer is not None else {}
        return Response(response_data, status=status.HTTP_200_OK)


class WorkflowView(WorkflowViewMixin, BaseViewSet):

    @classonlymethod
    def as_view(cls, actions=None, **initkwargs):
        actions = actions or {'post': 'perform_action'}
        return super().as_view(actions, **initkwargs)

    @classonlymethod
    def as_form_view(cls, **initkwargs):
        """
        Шорткат для формы workflow
        """
        return cls.as_view(
            actions={'get': 'update_form'},
            **initkwargs
        )

    def perform_action(self, request, *args, **kwargs):
        return super(WorkflowView, self).perform_action(self.action_name)


def unfold_query_params(query_params, list_fields=None):
    params = {}
    list_fields = list_fields or []
    for field_name in query_params:
        if field_name in list_fields:
            params[field_name] = query_params.getlist(field_name)
        else:
            params[field_name] = query_params.get(field_name)
    return params
