import sform
from django.core.exceptions import ValidationError
from rest_framework.response import Response

from ok.api.approvements.forms import ApprovementParentStageForm, JsonField
from ok.api.core.views import BaseViewSet
from ok.tracker.controllers import TRIGGER_CREATE_TYPE, create_trigger
from ok.flows.models import Flow


class TriggerForm(sform.SForm):
    name = sform.CharField(state=sform.REQUIRED)
    queue = sform.CharField(state=sform.REQUIRED)
    groups = sform.GridField(
        field_instance=sform.CharField(
            max_length=255,
            state=sform.REQUIRED,
        ),
        state=sform.REQUIRED,
    )
    trigger_create_type = sform.ChoiceField(
        choices=TRIGGER_CREATE_TYPE,
        state=sform.REQUIRED,
    )
    is_parallel = sform.BooleanField(default=False)

    oauth_token = sform.CharField()
    stages = sform.GridField(
        field_instance=sform.FieldsetField(ApprovementParentStageForm),
    )
    flow_name = sform.CharField()
    flow_context = JsonField()

    def clean(self):
        cleaned_data = super().clean()
        if cleaned_data['trigger_create_type'] == TRIGGER_CREATE_TYPE.complete:
            if not cleaned_data['oauth_token']:
                raise ValidationError(
                    'Required oauth_token if trigger_create_type is complete',
                    code='oauth_token_required',
                )

        return cleaned_data

    def clean_flow_name(self, flow_name):
        if flow_name and not Flow.objects.filter(name=flow_name).exists():
            raise ValidationError('Flow not found', code='flow_not_found')
        return flow_name


class TriggerView(BaseViewSet):
    validator_class = TriggerForm

    def post(self, request, *args, **kwargs):
        validator = self.get_validator_object(request.data)
        self.validate(validator)

        url = create_trigger(**validator.cleaned_data)

        return Response({'trigger_url': url})
