import waffle

from collections import defaultdict, deque
from datetime import timedelta
from itertools import count
from logging import getLogger
from typing import Dict

from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.db import models, transaction
from django.utils import timezone

from ok.api.core.errors import EditException, FlowDroppedHttp200, TableFlowCallFailed
from ok.approvements.choices import (
    APPROVEMENT_STATUSES,
    APPROVEMENT_RESOLUTIONS,
    APPROVEMENT_HISTORY_EVENTS,
    APPROVEMENT_STAGE_STATUSES,
)
from ok.approvements.models import (
    Approvement,
    ApprovementStage,
    create_history_entries,
    ApprovementHistory,
    ApprovementGroup,
)
from ok.approvements.tasks import (
    update_approvement_issue_task,
    set_approvement_tracker_comment_id_task,
    ping_approvers_task,
    send_callback_task,
    close_or_restore_approvement_task,
)
from ok.approvements.tracker import IssueFieldEnum
from ok.flows.executor import execute_flow
from ok.notifications.approvements import (
    ApprovementApprovedByResponsibleNotification,
    ApprovementFinishedNotification,
    ApprovementRequiredNotification,
    ApprovementCancelledNotification,
    ApprovementSuspendedNotification,
    ApprovementReminderNotification,
    ApprovementOverdueNotification,
)
from ok.staff.models import Group
from ok.staff.tasks import sync_group_memberships_task
from ok.tracker.issues import get_issue_author
from ok.tracker.queues import get_or_create_queue_by_object_id
from ok.utils.cache import memoize
from ok.utils.calendar import get_holidays
from ok.utils.context import request_context
from ok.utils.gap import get_gaps
from ok.utils.http_session import requests_retry_session
from ok.utils.lock import lock_by_approvement
from ok.utils.staff import (
    get_staff_users_iter,
    get_staff_groupmembership_iter,
)
from ok.utils.tvm import get_service_ticket, TVM2Error


logger = getLogger(__name__)


class NoStagesForFlowException(Exception):
    pass


class ExceededApproversLimit(Exception):
    pass


class ApprovementController:

    def __init__(self, instance):
        self.instance = instance

    @classmethod
    @transaction.atomic()
    def create(cls, data: Dict, initiator):
        flow_name = data.pop('flow_name', None)
        flow_context = data.pop('flow_context', None)
        stages = data.pop('stages', None)
        create_comment = data.pop('create_comment')
        if flow_name:
            request_context.set('flow_name', flow_name)
            result = cls.get_data_from_flow(flow_name, flow_context)
            stages = result['data'].pop('stages', None) or stages
            approve_if_no_approvers = result['data'].pop('approve_if_no_approvers', None)
            if not stages:
                if result['detail'].get('error') == 'Flow dropped':
                    raise FlowDroppedHttp200()
                elif result['detail'].get('error') == 'Table flow call failed':
                    raise TableFlowCallFailed(result.get('detail'))
                elif approve_if_no_approvers:
                    data['status'] = APPROVEMENT_STATUSES.closed
                    data['resolution'] = APPROVEMENT_RESOLUTIONS.approved
                    stages = []
                else:
                    raise NoStagesForFlowException()
            data.update(result['data'])

        data['tracker_queue'] = get_or_create_queue_by_object_id(data['object_id'])

        approvement = Approvement.objects.create(**data)

        stages = cls.normalize_stages(stages)
        # Создаём стадии балком – по одному запросу на каждый слой
        stage_layers = cls.prepare_stage_layers(approvement, stages)
        for layer in sorted(stage_layers):
            # Когда мы собираем слои, у родителей ещё нет id,
            # и в parent_id всегда проставляется None.
            # Поэтому явно проставляем parent ещё раз после того,
            # как мы уже записали верхние слои в БД
            for stage in stage_layers[layer]:
                stage.parent = stage.parent
            ApprovementStage.objects.bulk_create(stage_layers[layer])

        ping_task = ping_approvers_task.si(approvement.id, initiator)
        if approvement.is_tracker_approvement:
            # Сначала пытаемся получить id коммента в Трекере, а потом пингуем согласующих
            tasks_chain = set_approvement_tracker_comment_id_task.si(approvement.id) | ping_task
            tasks_chain.delay()

            update_approvement_issue_task.delay(
                approvement_id=approvement.id,
                fields=[
                    IssueFieldEnum.APPROVEMENT_STATUS,
                    IssueFieldEnum.ACCESS,
                    IssueFieldEnum.APPROVERS,
                    IssueFieldEnum.CURRENT_APPROVERS,
                ],
            )
        else:
            ping_task.delay(approvement.id, initiator)
        if create_comment:
            # В данном случае согласование создается с помощью механизма восстановления согласования
            close_or_restore_approvement_task.delay(approvement.id, None, create_comment=True)

        # Сохраняем группы координаторов и запускаем для них синхронизацию участников
        group_urls = data.get('groups') or []
        group_urls = list({url for url in group_urls if url})
        groups = [Group(url=url) for url in group_urls]
        approvement_groups = [
            ApprovementGroup(approvement=approvement, group_id=url)
            for url in group_urls
        ]
        Group.objects.bulk_create(groups, ignore_conflicts=True)
        ApprovementGroup.objects.bulk_create(approvement_groups)
        sync_group_memberships_task.delay(group_urls)

        return approvement

    def update(self, data: Dict, initiator):
        if 'stages' in data:
            stages = data.pop('stages')
            cur_stages = list(self.instance.stages.filter(parent__isnull=True))
            updated_stages = set()

            if len(stages) != len(cur_stages):
                raise EditException("Can't add or delete stages")

            for cur, new in zip(cur_stages, stages):
                if self.is_stages_equal(cur, new):
                    continue

                if not cur.is_active:
                    raise EditException('Cannot edit not active stage')

                if (cur.is_parent and new['approver']) or (not cur.is_parent and not new['approver']):
                    raise EditException('Edit error')
                elif not cur.is_parent:
                    cur.approver = new['approver']
                    updated_stages.add(cur)
                elif cur.stages:
                    updated = self.merge_stages(cur, new)
                    updated_stages |= updated

            ApprovementStage.objects.bulk_update(updated_stages, ['approver'])

            create_history_entries(
                objects=updated_stages,
                event=APPROVEMENT_HISTORY_EVENTS.approver_changed,
                user=initiator,
            )

        for key in data:
            if not hasattr(self.instance, key) or data[key] != getattr(self.instance, key):
                setattr(self.instance, key, data[key])

        self.instance.save()

        return self.instance

    @classmethod
    def merge_stages(cls, old, new):
        updated_stages = set()

        cur_stages = old.stages.all()

        if len(new['stages']) != len(cur_stages):
            raise EditException("Can't add or delete stages")

        for cur, new in zip(cur_stages, new['stages']):
            if cur.approver != new['approver']:
                if not cur.is_active:
                    raise EditException('Cannot edit not active stage')
                cur.approver = new['approver']
                updated_stages.add(cur)

        return updated_stages

    @classmethod
    def is_stages_equal(cls, old, new):
        if not old.is_parent and new['stages']:
            return False

        old_stages = list(old.stages.all())

        if len(old_stages) != len(new['stages']):
            return False

        for sub1, sub2 in zip(list(old.stages.all()), new['stages']):
            if sub1.approver != sub2['approver']:
                return False

        return (old.approver == new['approver']
                and old.need_approvals == new['need_approvals'])

    @classmethod
    def normalize_stages(cls, stages):
        res_stages = []

        for stage in stages:
            normalized_stage = cls._normalize_stage(stage)

            if normalized_stage:
                normalized_stage['need_approvals'] = min(
                    stage.get('need_approvals', 1),
                    len(normalized_stage.get('stages', []))
                )
                res_stages.append(normalized_stage)

        return res_stages

    @classmethod
    def _normalize_stage(cls, stage):
        if not stage.get('approver', '') and not stage.get('stages', []):
            return None

        if 'stages' in stage and stage['stages']:
            res_stage = []

            for sub_stage in stage['stages']:
                normalized_stage = cls._normalize_stage(sub_stage)

                if normalized_stage:
                    res_stage.append(normalized_stage)

            stage['stages'] = res_stage

            if len(stage['stages']) == 1:
                return stage['stages'][0]

            if len(stage['stages']) == 0:
                return None

        return stage

    @classmethod
    def get_data_from_flow(cls, flow_name, flow_context) -> Dict:
        return execute_flow(flow_name, flow_context)

    @classmethod
    def prepare_stage_layers(cls, approvement, stages, parent=None, position_counter=None,
                             layer=0, result=None, status=APPROVEMENT_STAGE_STATUSES.pending):
        position_counter = position_counter or count()
        result = result or defaultdict(list)

        for stage_data in stages:
            children = stage_data.pop('stages', [])
            position = next(position_counter)
            if position >= settings.APPROVEMENT_STAGES_LIMIT:
                raise ExceededApproversLimit

            # Ставим статус `current` для всех стадий в параллельном согласовании,
            # а также для 0-вой стадии и её потомков
            stage_status = status
            if approvement.is_parallel or position == 0:
                stage_status = APPROVEMENT_STAGE_STATUSES.current

            stage = ApprovementStage(
                approvement=approvement,
                parent=parent,
                position=position,
                status=stage_status,
                **stage_data,
            )
            result[layer].append(stage)
            cls.prepare_stage_layers(
                approvement=approvement,
                stages=children,
                parent=stage,
                position_counter=position_counter,
                layer=layer + 1,
                result=result,
                status=stage_status,
            )

        return result

    def perform_auto_approve(self, approvement_source=''):
        if not self.instance.is_auto_approving or not self.instance.is_tracker_approvement:
            return self.instance

        ticket_author = get_issue_author(self.instance.object_id)
        current_stages = self.instance.current_stages

        for stage in current_stages:
            if stage.approver == ticket_author:
                self._approve_stages([stage], ticket_author, approvement_source)

        return self.instance

    def ping_approvers(self, initiator):
        # Для статистики событие пинга записываем ещё и в родительскую стадию,
        # чтобы было удобно для неё вычислять длительность
        pinged_stages = self.instance.current_stages
        pinged_stages += [s.parent for s in pinged_stages if s.parent]

        create_history_entries(
            objects=set(pinged_stages),
            event=APPROVEMENT_HISTORY_EVENTS.ping_sent,
            user=initiator,
        )
        ApprovementRequiredNotification(
            instance=self.instance,
            initiator=self.instance.author,
            current_stages=self.instance.current_stages,
        ).send()

    @lock_by_approvement
    def approve(self, stages, initiator, approvement_source=''):
        is_current_approved = self._approve_stages(stages, initiator, approvement_source)
        # Уведомляем следующего, только если завершена текущая корневая стадия,
        # и согласование последовательное
        notify_next = is_current_approved and not self.instance.is_parallel
        self.on_after_approve(initiator, notify_next)
        return self.instance

    def on_after_approve(self, initiator, notify_next):
        issue_update_fields = [IssueFieldEnum.CURRENT_APPROVERS]
        is_finished = not (
            self.instance.stages
            .filter(parent_id__isnull=True)
            .exclude(status='approved')
            .exists()
        )
        if is_finished:
            self.instance.status = APPROVEMENT_STATUSES.closed
            self.instance.resolution = APPROVEMENT_RESOLUTIONS.approved
            self.instance.save()
            ApprovementFinishedNotification(
                instance=self.instance,
                initiator=initiator,
            ).send()
            issue_update_fields.append(IssueFieldEnum.APPROVEMENT_STATUS)
            self.send_callback()

        elif notify_next:
            self.ping_approvers(initiator)

        self._update_approvement_issue(issue_update_fields)

    def send_callback(self):
        send_callback_task.delay(self.instance.id)

    def _approve_stages(self, stages, initiator, approvement_source) -> bool:
        """
        :return: признак того, что была согласована текущая корневая стадия
        """
        current_root_stage = None
        active_stages = self.instance.stages.active().select_related('parent')
        already_approved_stages = self.instance.stages.filter(
            status=APPROVEMENT_STAGE_STATUSES.approved,
            parent__need_approvals__isnull=False
        ).select_related('parent')
        approved_stages = set()
        cancelled_stages = set()
        update_data = dict(
            is_approved=True,
            status=APPROVEMENT_STAGE_STATUSES.approved,
            approved_by=initiator,
            approvement_source=approvement_source,
            modified=timezone.now(),
        )
        approved_by_responsible_receivers = set()

        # Строим простое дерево активных стадий {parent: set(children)}
        active_tree = defaultdict(set)
        for stage in active_stages:
            active_tree[stage.parent].add(stage)
            if not current_root_stage and not stage.parent:
                current_root_stage = stage

        approve_tree = defaultdict(set)
        for stage in already_approved_stages:
            approve_tree[stage.parent].add(stage)

        queue = deque(stages)
        queued = set(stages)

        # Окаем все возможные стадии
        while queue:
            stage = queue.popleft()
            for field, value in update_data.items():
                setattr(stage, field, value)
            approved_stages.add(stage)
            approve_tree[stage.parent].add(stage)
            if stage.approver and stage.approver != stage.approved_by:
                approved_by_responsible_receivers.add(stage.approver)
            active_tree[stage.parent].discard(stage)
            if not stage.parent:
                continue
            is_parent_approved = (
                not active_tree[stage.parent] or
                len(approve_tree[stage.parent]) >= (stage.parent.need_approvals or float('inf'))
            )
            if is_parent_approved and stage.parent not in queued:
                queue.append(stage.parent)
                queued.add(stage.parent)

        # Отменяем стадии, которые больше неактуальны
        for parent in active_tree:
            if active_tree[parent] and parent not in approved_stages:
                continue
            for stage in active_tree[parent]:
                stage.status = APPROVEMENT_STAGE_STATUSES.cancelled
                stage.modified = update_data['modified']
                cancelled_stages.add(stage)
                if waffle.switch_is_active('enable_complex_stage_approved_by_notification'):
                    approved_by_responsible_receivers.add(stage.approver)

        # Сохраняем все изменения в БД
        changed_stages = approved_stages | cancelled_stages
        ApprovementStage.objects.bulk_update(changed_stages, update_data.keys())
        create_history_entries(
            objects=changed_stages,
            event=APPROVEMENT_HISTORY_EVENTS.status_changed,
            user=initiator,
        )

        if approved_by_responsible_receivers:
            ApprovementApprovedByResponsibleNotification(
                instance=self.instance,
                initiator=initiator,
                receivers=approved_by_responsible_receivers,
            ).send()

        is_current_approved = current_root_stage in approved_stages
        if is_current_approved:
            self.change_stages_status(
                stages=self.instance.next_stages,
                status=APPROVEMENT_STAGE_STATUSES.current,
                initiator=initiator,
            )

        self.perform_auto_approve(approvement_source=approvement_source)
        return is_current_approved

    def change_stages_status(self, stages, status, initiator, **extra_update_params):
        stage_id_list = list(stages.values_list('id', flat=True))
        stages.update(
            status=status,
            modified=timezone.now(),
            **extra_update_params,
        )
        create_history_entries(
            objects=ApprovementStage.objects.filter(id__in=stage_id_list),
            event=APPROVEMENT_HISTORY_EVENTS.status_changed,
            user=initiator,
        )

    @lock_by_approvement
    def suspend(self, initiator, status=APPROVEMENT_STATUSES.suspended, current_stages=None):
        current_stages = self.instance.current_stages if current_stages is None else current_stages
        self.instance.status = status
        self.instance.save()
        self.change_stages_status(
            stages=self.instance.stages.active(),
            status=APPROVEMENT_STAGE_STATUSES.suspended,
            initiator=initiator,
        )

        if waffle.switch_is_active(f'enable_approvement_{status}_notification'):
            ApprovementSuspendedNotification(
                instance=self.instance,
                initiator=initiator,
                current_stages=current_stages,
            ).send()

        self._update_approvement_issue([IssueFieldEnum.APPROVEMENT_STATUS])

        return self.instance

    @lock_by_approvement
    def reject(self, stage, initiator, disapproval_reason=None):
        self.instance.chosen_disapproval_reason = disapproval_reason
        self.instance.save()
        current_stages = self.instance.current_stages
        stage.is_approved = False
        stage.status = APPROVEMENT_STAGE_STATUSES.rejected
        stage.save(update_fields=['is_approved', 'status', 'modified'])
        return self.suspend(initiator, APPROVEMENT_STATUSES.rejected, current_stages, lock=False)

    @lock_by_approvement
    def resume(self, initiator):
        self.instance.status = APPROVEMENT_STATUSES.in_progress
        self.instance.save()
        common_data = {
            'initiator': initiator,
            'is_approved': None,
        }
        self.change_stages_status(
            stages=self.instance.next_stages,
            status=APPROVEMENT_STAGE_STATUSES.current,
            **common_data,
        )
        self.change_stages_status(
            stages=self.instance.stages.filter(status__in=(
                APPROVEMENT_STAGE_STATUSES.suspended,
                APPROVEMENT_STAGE_STATUSES.rejected,
            )),
            status=APPROVEMENT_STAGE_STATUSES.pending,
            **common_data,
        )
        self.ping_approvers(initiator)

        self._update_approvement_issue([IssueFieldEnum.APPROVEMENT_STATUS])
        return self.instance

    @lock_by_approvement
    def close(self, initiator):
        current_stages = self.instance.current_stages
        self.instance.status = APPROVEMENT_STATUSES.closed
        self.instance.resolution = APPROVEMENT_RESOLUTIONS.declined
        self.instance.save()
        self.change_stages_status(
            stages=self.instance.stages.filter(status__in=(
                APPROVEMENT_STAGE_STATUSES.suspended,
                APPROVEMENT_STAGE_STATUSES.pending,
                APPROVEMENT_STAGE_STATUSES.current,
            )),
            status=APPROVEMENT_STAGE_STATUSES.cancelled,
            initiator=initiator,
        )

        if waffle.switch_is_active('enable_approvement_cancelled_notification'):
            ApprovementCancelledNotification(
                instance=self.instance,
                initiator=initiator,
                current_stages=current_stages,
            ).send()

        self._update_approvement_issue([IssueFieldEnum.APPROVEMENT_STATUS])

        return self.instance

    def _update_approvement_issue(self, fields):
        if self.instance.is_tracker_approvement:
            update_approvement_issue_task.delay(self.instance.id, fields)


def send_callback(approvement):
    if not approvement.callback_url:
        return

    from ok.api.approvements.serializers import ApprovementSerializer
    data = ApprovementSerializer(approvement).data
    headers = {}
    if approvement.tvm_id:
        try:
            ticket = get_service_ticket(approvement.tvm_id)
        except TVM2Error:
            ticket = None

        if ticket:
            headers['X-Ya-Service-Ticket'] = ticket

    try:
        session = requests_retry_session(retries=5)
        answer = session.post(url=approvement.callback_url, json=data, headers=headers)
    except Exception:
        logger.error('Error sending callback for %s url', approvement.callback_url)
        raise

    logger.info(
        'Callback to %s of %s approvement returned status code %s',
        approvement.callback_url,
        approvement.uuid,
        answer.status_code,
    )
    logger.info('Response: %s', answer.content[:100])


def get_validated_staff_users(logins):
    login_to_user_map = get_validated_map_staff_users(logins)
    # Восстанавливаем изначальный порядок пользователей
    return [
        login_to_user_map[login]
        for login in logins
        if login in login_to_user_map
    ]


def get_validated_map_staff_users(logins):
    fields = [
        'login',
        'name.first.ru',
        'name.last.ru',
        'official.affiliation',
    ]
    users = get_staff_users_iter({
        'login': ','.join(logins),
        'official.is_dismissed': False,
        '_fields': ','.join(fields),
    })

    return {
        user['login']: {
            'login': user['login'],
            'fullname': '{} {}'.format(
                user['name']['first']['ru'],
                user['name']['last']['ru'],
            ),
            'affiliation': user['official']['affiliation'],
        }
        for user in users
    }


def fill_staff_users(stages):
    logins = []
    for stage in stages:
        if stage.get('stages'):
            logins += [
                sub_stage['approver']
                for sub_stage in stage['stages']
                if sub_stage.get('approver')
            ]
        elif stage.get('approver'):
            logins.append(stage['approver'])
    staff_users = get_validated_map_staff_users(logins)
    res = []
    for stage in stages:
        if 'stages' in stage:
            cur_stages = [
                {'approver': staff_users[sub_stage['approver']]}
                for sub_stage in stage['stages']
                if sub_stage.get('approver') in staff_users
            ]

            res.append(
                {
                    'stages': cur_stages,
                    'need_all': stage.get('need_approvals', 1) == len(cur_stages),
                    'need_approvals': stage.get('need_approvals', 1),
                }
            )
        elif stage.get('approver') in staff_users:
            res.append({'approver': staff_users[stage['approver']]})
    return res


def get_staff_user_logins(logins):
    users = get_staff_users_iter({
        'login': ','.join(logins),
        'official.is_dismissed': False,
        '_fields': 'login',
    })
    return {user['login'] for user in users}


@memoize(600)
def get_staff_group_member_logins(groups):
    memberships = get_staff_groupmembership_iter({
        'group.url': ','.join(groups),
        'person.official.is_dismissed': False,
        '_fields': 'person.login',
    })
    return {membership['person']['login'] for membership in memberships}


def _get_active_not_pinged_today_stages_qs():
    """
    Не шлём напоминания тем, кого сегодня уже пинговали.
    Например, если стадия сегодня только началась.
    """
    approvement_stage_content_type = ContentType.objects.get_for_model(ApprovementStage)
    today = timezone.now().replace(hour=0, minute=0, second=0, microsecond=0)

    stages_history = (
        ApprovementHistory.objects
        .filter(
            content_type=approvement_stage_content_type,
            object_id=models.OuterRef('id')
        )
        .order_by()
    )
    last_ping_date_subquery = models.Subquery(
        queryset=(
            stages_history
            .filter(event=APPROVEMENT_HISTORY_EVENTS.ping_sent)
            .values('object_id')
            .annotate(last_ping_date=models.Max('created'))
            .values('last_ping_date')
        ),
        output_field=models.DateTimeField(),
    )
    has_question = models.Exists(
        stages_history
        .filter(event=APPROVEMENT_HISTORY_EVENTS.question_asked)
        .values('id')
    )

    return (
        ApprovementStage.objects
        .active()
        .exclude(approver='')
        .annotate(last_ping_date=last_ping_date_subquery, has_question=has_question)
        # Предполагаем, что текущие статидии не могли не получить запись о пинге в историю
        .filter(
            last_ping_date__isnull=False,
            last_ping_date__lt=today,
            has_question=False,
        )
    )


def _get_active_approvements_with_current_stages_only():
    active_stages_prefetch = models.Prefetch(
        lookup='stages',
        queryset=_get_active_not_pinged_today_stages_qs(),
        to_attr='active_stages',
    )
    active_approvements = (
        Approvement.objects
        .filter(status=APPROVEMENT_STATUSES.in_progress)
        .prefetch_related(active_stages_prefetch)
    )
    # Предполагаем, что статидии, идущие за текущими, не могли получить запись о пинге в историю
    return active_approvements


def _get_active_approver_to_approvement_stages_map(active_approvements):
    approver_to_approvement_stages_map = defaultdict(list)
    for approvement in active_approvements:
        for stage in approvement.active_stages:
            approver_to_approvement_stages_map[stage.approver].append(stage)
    return approver_to_approvement_stages_map


def get_absent_approvers(approvers, date):
    result = set()

    for gap in get_gaps(approvers, date, date, workflow='vacation'):
        logger.info('Approver %s has a vacation: %s', gap['person_login'], gap)
        result.add(gap['person_login'])

    for user, holidays in get_holidays(approvers, date, date).items():
        if holidays:
            logger.info('Approver %s has holidays: %s', user, holidays)
            result.add(user)

    return result


def notify_current_approvers():
    active_approvements = _get_active_approvements_with_current_stages_only()
    receivers_map = _get_active_approver_to_approvement_stages_map(active_approvements)

    try:
        absent_approvers = get_absent_approvers(receivers_map.keys(), timezone.now().date())
    except Exception:
        if not waffle.switch_is_active('skip_gap_errors_on_notify'):
            raise
        logger.warning('Cannot get absence', exc_info=True)
        absent_approvers = set()

    reminded_stages = []

    for approver, approvement_stages in receivers_map.items():
        if approver in absent_approvers:
            logger.info('Skip notification for %s because of absence', approver)
            continue

        approvements = [stage.approvement for stage in approvement_stages]

        notification = ApprovementReminderNotification(
            receiver=approver,
            approvements=approvements,
        )
        notification.send()
        reminded_stages.extend(approvement_stages)

    create_history_entries(
        objects=reminded_stages,
        event=APPROVEMENT_HISTORY_EVENTS.ping_sent,
        # Это регулярные напоминания, у них нет инициатора
        user=None,
    )


def _get_receiver_to_overdue_approvement_map():
    # Выбираем все ещё не подтвержденные стадии активных согласований,
    # первый пинг по которым был больше APPROVEMENT_STAGE_OVERDUE_DAYS дней назад
    overdue_ping_time = timezone.now() - timedelta(settings.APPROVEMENT_STAGE_OVERDUE_DAYS)
    overdue_stages = (
        ApprovementStage.objects
        .active()
        .filter(
            approvement__status=APPROVEMENT_STATUSES.in_progress,
            history__event=APPROVEMENT_HISTORY_EVENTS.ping_sent,
        )
        .annotate(first_ping_time=models.Min('history__created'))
        .filter(first_ping_time__lt=overdue_ping_time)
        .select_related('approvement')
    )

    receiver_to_overdue_approvement_map = defaultdict(lambda: defaultdict(list))
    for stage in overdue_stages:
        approvement = stage.approvement
        receivers = approvement.info_mails_to or [approvement.author]
        for reviever in receivers:
            receiver_approvements = receiver_to_overdue_approvement_map[reviever]
            receiver_approvements[approvement].append(stage)
    return receiver_to_overdue_approvement_map


def notify_overdue_approvement_receivers():
    receivers_map = _get_receiver_to_overdue_approvement_map()
    for receiver, approvement_to_current_stages in receivers_map.items():
        notification = ApprovementOverdueNotification(
            receiver=receiver,
            approvement_to_current_stages=approvement_to_current_stages,
        )
        notification.send()
