from collections import defaultdict
from datetime import timedelta
from operator import attrgetter

from django.utils import timezone

from ok.approvements.choices import (
    APPROVEMENT_STATUSES,
    APPROVEMENT_HISTORY_EVENTS,
    APPROVEMENT_STAGE_STATUSES,
)


def calculate_durations(events):
    result = defaultdict(timedelta)
    if not events:
        return result

    events_iter = iter(events)
    prev_time, prev_status = next(events_iter)

    for time, status in events_iter:
        duration = time - prev_time
        result['total'] += duration
        result[prev_status] += duration

        prev_time = time
        prev_status = status

    return result


def get_approvement_stats(approvement):
    events = list(
        approvement.history
        .filter(event=APPROVEMENT_HISTORY_EVENTS.status_changed)
        .values_list('created', 'status')
    )
    # Для незакрытого согласования статистику нужно считать относительно текущего времени
    if approvement.status != APPROVEMENT_STATUSES.closed:
        events.append((timezone.now(), None))

    durations = calculate_durations(events)

    # Note: считаем, что время, на которое согласование приостанавливалось
    # – это время пребывания согласования в статусах suspended и rejected
    suspense_duration = (
        durations[APPROVEMENT_STATUSES.suspended]
        + durations[APPROVEMENT_STATUSES.rejected]
    )

    return {
        'duration': {
            'total': int(durations['total'].total_seconds()),
            'active': int(durations[APPROVEMENT_STATUSES.in_progress].total_seconds()),
            'suspended': int(suspense_duration.total_seconds()),
        },
    }


def get_approvement_stage_stats(stage):
    first_events = defaultdict(dict)
    events = []

    for event in stage.history.all():
        first_events.setdefault(event.event, {'any_status': event})
        first_events[event.event].setdefault(event.status, event)

        # начинаем собирать все события для рассчета длительности только с момента первого пинга
        if event.event == APPROVEMENT_HISTORY_EVENTS.ping_sent or events:
            events.append((event.created, event.status))

    first_status_change_events = first_events[APPROVEMENT_HISTORY_EVENTS.status_changed]

    ping_event = first_events[APPROVEMENT_HISTORY_EVENTS.ping_sent].get('any_status')
    approve_event = first_status_change_events.get(APPROVEMENT_STAGE_STATUSES.approved)
    cancelled_event = first_status_change_events.get(APPROVEMENT_STAGE_STATUSES.cancelled)
    reject_event = first_status_change_events.get(APPROVEMENT_STAGE_STATUSES.rejected)
    question_event = first_events[APPROVEMENT_HISTORY_EVENTS.question_asked].get('any_status')

    reaction_events = filter(None, (
        question_event,
        reject_event,
        stage.approved_by == stage.approver and approve_event,
    ))
    reaction_event = min(reaction_events, default=None, key=attrgetter('created'))

    if not approve_event and not cancelled_event:
        events.append((timezone.now(), APPROVEMENT_STAGE_STATUSES.pending))
    durations = calculate_durations(events)

    # Note: считаем, что время, на которое стадия приостанавливалась
    # – это время пребывания стадии в статусах suspended и rejected.
    # Активное время – в статусах pending и current.
    suspense_duration = (
        durations[APPROVEMENT_STAGE_STATUSES.suspended]
        + durations[APPROVEMENT_STAGE_STATUSES.rejected]
    )
    activity_duration = (
        durations[APPROVEMENT_STAGE_STATUSES.pending]
        + durations[APPROVEMENT_STAGE_STATUSES.current]
    )

    return {
        'time': {
            'ping': ping_event.created if ping_event else None,
            'reaction': reaction_event.created if reaction_event else None,
            'approve': approve_event.created if approve_event else None,
            'reject': reject_event.created if reject_event else None,
        },
        'duration': {
            'total': int(durations['total'].total_seconds()),
            'active': int(activity_duration.total_seconds()),
            'suspended': int(suspense_duration.total_seconds()),
        },
    }
