import logging
from datetime import timedelta

import waffle
from django.utils import timezone, translation
from startrek_client.exceptions import NotFound, StartrekError
from ylog.context import log_context

from ok.approvements.choices import APPROVEMENT_STATUSES, APPROVEMENT_TYPES
from ok.approvements.models import Approvement, ApprovementHistory
from ok.approvements.tracker import (
    set_approvement_tracker_comment_id,
    IssueApprovementSerializer,
)
from ok.approvements.yt.tables import YTApprovementHistoryTable
from ok.celery_app import app
from ok.tracker.base import client as tracker
from ok.utils import staff
from ok.utils.datetime import get_beginning_of_moscow_day

logger = logging.getLogger(__name__)


@app.autoretry_task(max_retries=3)
def ping_approvers_task(approvement_id, initiator):
    from ok.approvements.controllers import ApprovementController
    approvement = Approvement.objects.get(id=approvement_id)
    ctl = ApprovementController(approvement)
    ctl.ping_approvers(initiator)


@app.autoretry_task(max_retries=3)
def update_approvement_issue_task(approvement_id, fields):
    """
    Обновляет в тикете согласования поля fields
    """
    approvement = Approvement.objects.get(id=approvement_id)
    if not approvement.is_tracker_approvement:
        logger.info('Skip update_approvement_issue_task for non tracker approvement')
        return

    fields = IssueApprovementSerializer(approvement, fields=fields).data

    try:
        issue = tracker.issues[approvement.object_id]
        issue.update(**fields)
    except Exception:
        with log_context(approvement=approvement.id):
            logger.error('Could not update issue %s: %s', approvement.object_id, fields)
        raise


@app.autoretry_task(max_retries=3, fail_silently=True)
def set_approvement_tracker_comment_id_task(approvement_id):
    approvement = Approvement.objects.get(id=approvement_id)
    set_approvement_tracker_comment_id(approvement)


@app.autoretry_task(max_retries=3, fail_silently=True)
def send_callback_task(approvement_id):
    from ok.approvements.controllers import send_callback
    approvement = Approvement.objects.get(id=approvement_id)
    send_callback(approvement)


@app.autoretry_task(max_retries=3)
def notify_overdue_approvement_authors_task():
    from ok.approvements.controllers import notify_overdue_approvement_receivers
    notify_overdue_approvement_receivers()


@app.autoretry_task(max_retries=3)
def notify_current_approvers_task():
    from ok.approvements.controllers import notify_current_approvers
    if waffle.switch_is_active('enable_reminders'):
        notify_current_approvers()


@app.autoretry_task(max_retries=3)
def save_approvement_history_in_yt():
    today = get_beginning_of_moscow_day(timezone.now())
    yesterday = today - timedelta(days=1)

    history_entries = ApprovementHistory.objects.filter(
        created__gte=yesterday,
        created__lt=today,
    )

    table = YTApprovementHistoryTable()
    table.write(history_entries)


def _close_or_restore_approvement(approvement, initiator, issue, is_comment_initial=False):
    from ok.approvements.workflow import ApprovementWorkflow
    from ok.tracker.controllers import restore_approvement
    if not initiator:
        if waffle.switch_is_active('enable_approvement_restore_if_unknown'):
            restore_approvement(issue, approvement, initiator, is_comment_initial)
        return

    wf = ApprovementWorkflow(approvement, initiator)
    close_action = wf.get_action('close')
    if close_action.is_available():
        close_action.perform()
    else:
        restore_approvement(issue, approvement, initiator, is_comment_initial)


@app.autoretry_task(max_retries=4)
def close_or_restore_approvement_task(approvement_id, initiator, old_comment_id=None, create_comment=False):
    approvement = Approvement.objects.get(id=approvement_id)
    author_lang = staff.get_user_language(approvement.author)
    translation.activate(author_lang or 'en')
    # Если нашли новый коммент, ничего не делаем
    if old_comment_id != approvement.tracker_comment_short_id:
        return
    issue = tracker.issues[approvement.object_id]
    _close_or_restore_approvement(approvement, initiator, issue, create_comment)


@app.task
def check_deleted_approvements():
    from ok.tracker.controllers import is_comment_removed, find_comment_remover
    approvements = (
        Approvement.objects
        .filter(type=APPROVEMENT_TYPES.tracker)
        .exclude(status=APPROVEMENT_STATUSES.closed)
        .exclude(object_id='')
    )
    if not waffle.switch_is_active('enable_approvement_restore_if_unknown'):
        approvements = approvements.filter(tracker_comment_short_id__isnull=False)
    for approvement in approvements:
        try:
            issue = tracker.issues[approvement.object_id]
        except NotFound:
            logger.warning('Issue not found: %s', approvement.object_id)
            continue
        except StartrekError:
            logger.warning('Could not get issue: %s', approvement.object_id)
            continue

        comment_id = approvement.tracker_comment_short_id
        if is_comment_removed(issue, comment_id):
            removed_by = find_comment_remover(issue, comment_id)
            tasks_chain = (
                # Сначала пытаемся найти другой коммент с этим согласованием,
                # а уже потом делаем всё остальное, если надо
                set_approvement_tracker_comment_id_task.si(approvement.id)
                | close_or_restore_approvement_task.si(
                    approvement_id=approvement.id,
                    initiator=removed_by,
                    old_comment_id=comment_id,
                )
            )
            tasks_chain.delay()
