import waffle
from django.utils.functional import cached_property
from ids.exceptions import BackendError

from ok.api.core.errors import Http400
from ok.approvements.choices import APPROVEMENT_STATUSES
from ok.approvements.controllers import (
    ApprovementController,
    get_staff_group_member_logins,
)
from ok.core.workflow import Action, Workflow


class ApprovementBaseAction(Action):

    @cached_property
    def ctl(self):
        return ApprovementController(self.instance)

    def has_permission(self):
        return self.wf.is_responsible


class ApproveAction(ApprovementBaseAction):

    valid_statuses = (
        APPROVEMENT_STATUSES.in_progress,
    )

    @cached_property
    def can_approve_as_responsible(self):
        return (
            self.wf.is_responsible
            and (
                not self.instance.is_complex_approvement
                or waffle.switch_is_active('enable_complex_approve_by_responsible')
            )
        )

    def has_permission(self):
        return self.can_approve_as_responsible or self.wf.is_active_approver

    def perform(self, approver, **params):
        # Note: Если пришёл простой согласующий (не из ответственных),
        # мы считаем, что он ставит "ок" за себя. Так как, только за себя
        # он и может ставить "ок". В идеале, вообще бы ругаться на такое,
        # но сейчас фронт не использует нашу ручку формы и всегда присылает
        # следующего по очереди согласующего
        if not self.can_approve_as_responsible:
            approver = self.user
            params.pop('stages', None)

        # Если это согласование по-новому (с выбором стадий)
        if params.get('stages'):
            stages = params['stages']

        # Если по-старому, то получаем стадию для конкретного согласующего
        else:
            stage = self.wf.get_stage(approver)
            stages = [stage] if stage else []

        # Если ответственный пытается согласовать уже согласованную стадию
        if not stages:
            self.raise_error('active_stage_not_found')

        approvement_source = params.get('approvement_source', '')
        return self.ctl.approve(stages, self.user, approvement_source=approvement_source)


class SuspendAction(ApprovementBaseAction):

    valid_statuses = (
        APPROVEMENT_STATUSES.in_progress,
    )

    def perform(self, **params):
        return self.ctl.suspend(self.user)


class RejectAction(ApprovementBaseAction):
    """
    Не ОК – действие, аналогичное приостановке,
    только выполняют его согласующие
    """
    valid_statuses = (
        APPROVEMENT_STATUSES.in_progress,
    )

    def has_permission(self):
        return self.instance.is_reject_allowed and self.wf.is_active_approver

    def perform(self, **params):
        stage = self.wf.get_stage(self.user)
        if not stage:
            self.raise_error('active_stage_not_found')
        disapproval_reason = params.get('disapproval_reason')
        if disapproval_reason:
            if disapproval_reason not in self.instance.disapproval_reasons:
                self.raise_error(code='Unavailable disapproval reason', error_class=Http400)
        return self.ctl.reject(stage, self.user, disapproval_reason)


class ResumeAction(ApprovementBaseAction):

    valid_statuses = (
        APPROVEMENT_STATUSES.suspended,
        APPROVEMENT_STATUSES.rejected,
    )

    def perform(self, **params):
        return self.ctl.resume(self.user)


class CloseAction(ApprovementBaseAction):

    valid_statuses = (
        APPROVEMENT_STATUSES.in_progress,
        APPROVEMENT_STATUSES.suspended,
        APPROVEMENT_STATUSES.rejected,
    )

    def perform(self, **params):
        return self.ctl.close(self.user)


class EditAction(ApprovementBaseAction):

    valid_statuses = (
        APPROVEMENT_STATUSES.in_progress,
    )

    def perform(self, **params):
        return self.ctl.update(params, self.user)


class ApprovementWorkflow(Workflow):

    @cached_property
    def is_author(self):
        return self.instance.author == self.user

    @cached_property
    def is_responsible(self):
        return self.is_author or self.user in self.group_members

    @cached_property
    def group_members(self):
        try:
            return (
                get_staff_group_member_logins(self.instance.groups)
                if self.instance.groups
                else set()
            )
        except BackendError:
            return set()

    @cached_property
    def is_active_approver(self):
        return self.get_stage(self.user) is not None

    def get_stage(self, approver):
        return (
            self.instance.stages
            .active()
            .filter(approver=approver)
            .first()
        )

    ACTION_MAP = {
        'approve': ApproveAction,
        'reject': RejectAction,
        'suspend': SuspendAction,
        'resume': ResumeAction,
        'close': CloseAction,
        'edit': EditAction,
    }
