# coding: utf-8
import functools
import logging
from uuid import uuid4

from celery import Celery as CeleryBase, Task as TaskBase
from celery.result import AsyncResult

from django.db import transaction
from django.conf import settings
from django_celery_monitoring.mixins import CachedChainTaskMixin
from django_tools_log_context.celery import CtxAwareMixin


logger = logging.getLogger(__name__)


def get_retry_countdown(retry):
    retry_periods = [5, 55, 240, 1500, 9000, 10800]
    return retry_periods[min(len(retry_periods) - 1, retry)]


class Task(CtxAwareMixin, CachedChainTaskMixin, TaskBase):

    def apply_async(self, args=None, kwargs=None, task_id=None, **options):
        # Если on_transaction_commit == True,
        # то таска будет выполнена только после коммита транзакции
        on_transaction_commit = options.pop('on_transaction_commit', True)
        task_id = task_id or str(uuid4())
        _apply_async = lambda: super(Task, self).apply_async(
            args=args,
            kwargs=kwargs,
            task_id=task_id,
            **options
        )
        if on_transaction_commit:
            transaction.on_commit(_apply_async)
        else:
            _apply_async()
        return AsyncResult(task_id)


class Celery(CeleryBase):

    task_cls = 'ok.celery_app:Task'

    def autoretry_task(self, *args_task, **opts_task):
        fail_silently = opts_task.pop('fail_silently', False)

        def decorator(func):
            original_task = super(Celery, self).task

            @original_task(*args_task, **opts_task)
            @functools.wraps(func)
            def wrapper(*args, **kwargs):
                try:
                    return func(*args, **kwargs)
                except Exception as exc:
                    is_last_retry = wrapper.request.retries == wrapper.max_retries
                    if is_last_retry and fail_silently:
                        logger.exception('Celery last retry failed %s', wrapper.name)
                        return
                    wrapper.retry(
                        countdown=get_retry_countdown(wrapper.request.retries),
                        exc=exc,
                        args=args,
                        kwargs=kwargs,
                    )
            return wrapper
        return decorator


app = Celery('ok')

app.config_from_object('django.conf:settings')
app.autodiscover_tasks(lambda: settings.INSTALLED_APPS)
