from django.core.exceptions import PermissionDenied
from ok.api.core.errors import Http400


class WorkflowError(Http400):
    """
    Нарушение воркфлоу
    """
    def __init__(self, code):
        super().__init__(code)


class Action:
    """
    Класс, описывающий действие над неким объектом instance, выполняемое
    пользователем с логином user.
    """
    valid_statuses = None
    status_field_name = 'status'

    def __init__(self, wf):
        self.wf = wf
        self.instance = self.wf.instance
        self.user = self.wf.user

    def is_status_correct(self):
        """
        Проверяем доступы, связанные непостредственно со статусами
        """
        if self.valid_statuses is None:
            return True
        return getattr(self.instance, self.status_field_name, None) in self.valid_statuses

    def has_permission(self):
        """
        Проверяем доступы, связанные с правами пользователя или другими полями объекта
        """
        return True

    def is_available(self):
        """
        Доступность действия
        """
        return self.is_status_correct() and self.has_permission()

    def perform(self, **params):
        # Совершает действие, независимо от ответа is_available.
        return self.instance

    def raise_error(self, code='', error_class=None):
        error_class = error_class or WorkflowError
        raise error_class(code)


class Workflow:
    """
    Точка входа для пула экшенов какой-то сущности.
    Содержит в себе глобальные данные, необходимые для каждого экшена.
    """
    ACTION_MAP = None

    def __init__(self, instance, user, **kwargs):
        self.instance = instance
        self.user = user
        self.kwargs = kwargs

    def get_action(self, action_name):
        if action_name not in self.ACTION_MAP:
            raise WorkflowError('action_does_not_exist')
        action_class = self.ACTION_MAP[action_name]
        return action_class(wf=self)

    def perform_action(self, action_name, **params):
        action = self.get_action(action_name)
        if not action.is_available():
            raise PermissionDenied
        return action.perform(**params)

    def get_actions(self):
        return {
            name: klass(wf=self).is_available()
            for name, klass in self.ACTION_MAP.items()
        }
