import logging

import datetime
import json
import traceback
from functools import partial

from ok.flows.models import Flow
from ok.flows.flow_functions import *
from ok.flows import flow_api

logger = logging.getLogger(__name__)


FLOW_FUNCTIONS = {
    'department_head': get_department_head,
    'department_chain_of_heads': get_department_chain_of_heads,
    'direction_head': get_direction_head,
    'department_chain_of_heads_under_direction': get_department_chain_of_heads_under_direction,
    'hr_partners': get_hr_partners,
    'department_url': get_department_url,
    'department_chain': get_department_chain,
}


def execute_flow(flow_name: str, context: Dict) -> Dict:
    with log_context(root_flow=flow_name):
        logger.info(f'Starting execute {flow_name}')
        return execute(flow_name, context)


def execute(flow_name, context, executions=None) -> Dict:
    is_root_execution = executions is None
    executions = executions or []
    data = {}
    try:
        flow = Flow.objects.get(name=flow_name)
    except Flow.DoesNotExist:
        raise FlowNotFound(flow_name)
    try:
        exec(flow.code, _make_flow_context(context, data, executions))
        flow.last_result = json.dumps(data)
    except FlowDropped:
        return {'data': {'stages': []}, 'detail': {'error': 'Flow dropped'}}
    except TableFlowExecutionError as e:
        return {'data': {'stages': []},
                'detail': {
                    'code': e.code,
                    'table_name': e.table_name,
                    'response': e.response,
                    'error': 'Table flow call failed',
                    }
                }
    except Exception:
        # Обрабатываем ошибку в корневом вызове
        if not is_root_execution:
            raise
        err = traceback.format_exc()
        logger.warning(
            f'Flow {flow_name} failed to complete at {datetime.datetime.now()} with trace:\n{err}'
        )
        flow.last_error = err
        data['stages'] = []
    try:
        # FIXME: can't save into database in GET requests due to app settings
        flow.save()
    except:
        pass
    return {'data': data, 'detail': {}}


def execute_code(code: str, context: Dict) -> Dict:
    data = {'stages': flow_api._Stages()}
    res = {'data': data, 'detail': {}, 'traceback': None}
    try:
        exec(code, _make_flow_context(context, data, []))
        res['data']['stages'] = data['stages']
    except FlowDropped:
        res['detail'] = {'error': 'Flow creation dropped'}
    except TableFlowExecutionError as e:
        res['detail'] = {'code': e.code,
                         'table_name': e.table_name,
                         'response': e.response,
                         'error': 'Table flow call failed',
                         }
    except Exception:
        res['traceback'] = traceback.format_exc()
    finally:
        return res


def execute_sub_flow(
        executions,
        data,
        whitelist,
        flow_name,
        **params,
) -> List[Dict[str, str]]:
    if len(executions) >= settings.FLOW_RECURSION_MAX_DEPTH:
        raise FlowRecursionError('Flow execution maximum recursion depth exceeded')
    executions.append((flow_name, params))

    with log_context(inner_flows=flow_name):
        logger.info(f'Starting execute subflow {flow_name}')
        res = execute(flow_name, params, executions=executions)['data']

    set_approvement_field(data, whitelist, **res)
    return res


def _make_flow_context(raw_context, data: Dict, executions):
    whitelist = [
        'is_reject_allowed',
        'is_auto_approving',
        'text',
        'is_parallel',
        'approve_if_no_approvers',
    ]

    if 'stages' not in data:
        data['stages'] = flow_api._Stages()

    flow_api.stages = data['stages']

    context = {
        'params': raw_context,
        'add_approver': partial(add_approver, data['stages']),
        'add_approvers_group': partial(add_approvers_group, data['stages']),
        'execute_flow': partial(execute_sub_flow, executions, data, whitelist),
        'get_persons_by_role_id': get_persons_by_role_id,
        'get_duty_by_service_slug': get_duty_by_service_slug,
        'extend_approvers': partial(extend_approvers, data['stages']),
        'remove_duplicates': partial(remove_duplicates, data['stages']),
        'drop_approvement': partial(drop_approvement, data['stages']),
        'call_table_flow': call_table_flow,
        'set_approvement_field': partial(set_approvement_field, data=data, whitelist=whitelist),
        'log': logger.info,
        'flow_api': flow_api,
    }
    context.update(FLOW_FUNCTIONS)
    return context
