from ok.utils.http_session import requests_retry_session
from ok.utils.abc_repo import worker
from django.conf import settings
from ok.utils.cache import memoize
from itertools import takewhile
from collections import namedtuple
from ok.utils.staff import get_staff_repo
from typing import List, Dict
from ylog.context import log_context

group_repo = get_staff_repo('group')
person_repo = get_staff_repo('person')

Department = namedtuple('Department', ['id', 'url', 'name', 'head', 'is_direction'])


@memoize()
def _get_departments_from_api(**staffapi_lookup) -> List[Department]:
    if not {'url', 'id', 'name'} & staffapi_lookup.keys():
        raise ValueError()  # TODO
    if 'id' in staffapi_lookup:
        # we use group api, where name and url are identical to dep
        # but id is not
        staffapi_lookup['department.id'] = staffapi_lookup.pop('id')

    fields = [
        'ancestors.department.id',
        'ancestors.department.kind',
        'ancestors.department.name',
        'ancestors.department.url',
        'ancestors.department.heads.person.login',
        'ancestors.department.heads.role',
        'department.id',
        'department.kind',
        'department.heads.person.login',
        'department.heads.role',
        'department.name',
        'department.url',
    ]
    lookup = {
        'type': 'department',
        '_fields': ','.join(fields),
    }
    lookup.update(staffapi_lookup)
    data = group_repo.get_one(lookup)

    groups_raw = [data] + list(reversed(data['ancestors']))
    departments_chain = []
    for group in groups_raw:
        dep = group['department']
        chief = None
        for head in dep['heads']:
            if head['role'] == 'chief':
                chief = head['person']['login']
                break
        is_direction = dep.get('kind') and dep['kind']['slug'] == 'direction'
        departments_chain.append(Department(
            id=dep['id'],
            url=dep['url'],
            name=dep['name']['full'],
            head=chief,
            is_direction=is_direction,
        ))
    return departments_chain


def get_departments_chain_with_heads(
    until_id=None,
    until_url=None,
    until_name=None,
    until_chief_login=None,
    include_until=True,
    **staffapi_lookup
) -> List[Department]:
    deps = _get_departments_from_api(**staffapi_lookup)

    until = until_id or until_name or until_url or until_chief_login
    if not until:
        return deps

    if until_id:
        is_not_until_dep = lambda dep: dep.id != until_id
    elif until_name:
        until_name = until_name.lower()
        is_not_until_dep = lambda dep: until_name not in {it.lower() for it in dep.name.values()}
    elif until_chief_login:
        is_not_until_dep = lambda dep: dep.head != until_chief_login
    else:
        is_not_until_dep = lambda dep: dep.url != until_url

    res = list(takewhile(is_not_until_dep, deps))
    if include_until and len(res) != len(deps):
        res.append(deps[len(res)])

    return res


def get_department_head(**kwargs):
    heads = get_department_chain_of_heads(**kwargs)
    return heads[0] if heads else None


def get_department_chain_of_heads(skip_duplicates=False, **kwargs):
    departments_chain = get_departments_chain_with_heads(**kwargs)
    heads = [dep.head for dep in departments_chain if dep.head]
    if skip_duplicates:
        heads_set = set()
        deduplicated_heads = []
        for head in reversed(heads):
            if head not in heads_set:
                deduplicated_heads.append(head)
                heads_set.add(head)
        heads = list(reversed(deduplicated_heads))
    return heads


def get_department_chain_of_heads_under_direction(**kwargs):
    departments_chain = get_departments_chain_with_heads(**kwargs)
    heads = []
    for dep in departments_chain:
        if dep.is_direction:
            break
        if dep.head:
            heads.append(dep.head)
    return heads


def get_direction_head(**kwargs):
    departments_chain = get_departments_chain_with_heads(**kwargs)
    direction = [dep for dep in departments_chain if dep.is_direction]
    if direction:
        return direction[0].head


@memoize()
def get_hr_partners(person_login=None, dep_url=None, dep_name=None, dep_id=None):
    if not person_login:
        person_login = get_department_head(url=dep_url, name=dep_name, id=dep_id)
    api_resp = person_repo.get_one({
        'login': person_login,
        '_fields': 'hr_partners.login',
    })
    return [it['login'] for it in api_resp['hr_partners']]


@memoize()
def get_department_url(person_login):
    api_resp = person_repo.get_one({
        'login': person_login,
        '_fields': 'department.url',
    })
    return api_resp['department_group']['department']['url']


@memoize()
def get_department_chain(person_login):
    api_resp = person_repo.get_one({
        'login': person_login,
        '_fields': 'department_group',
    })
    chain = [
        dep['department']['url']
        for dep in api_resp['department_group']['ancestors']
    ]
    chain.append(api_resp['department_group']['department']['url'])
    return chain


class TableFlowExecutionError(Exception):

    def __init__(self, table_name, code, response, *args, **kwargs):
        super(TableFlowExecutionError, self).__init__(*args, **kwargs)

        self.table_name = table_name
        self.code = code
        self.response = response


class FlowNotFound(Exception):

    def __init__(self, flow_name, *args, **kwargs):
        self.flow_name = flow_name
        super().__init__(*args, **kwargs)


class FlowRecursionError(Exception):
    pass


class FlowDropped(Exception):
    pass


def set_approvement_field(data, whitelist, **kwargs):
    for key in kwargs:
        if key in whitelist:
            data[key] = kwargs[key]


def add_approver(stages, approver):
    stages.append({'approver': str(approver)})


def add_approvers_group(stages, approvers, need_all=False):
    approvers = [{'approver': str(a)} for a in approvers]
    if approvers:
        stages.append({'need_approvals': len(approvers) if need_all else 1, 'stages': approvers})


def all_approvers_in_stages(stages):
    approvers = set()
    approvers |= {stage['approver'] for stage in stages if stage.get('approver')}
    for stage in stages:
        approvers |= {stage['approver'] for stage in stage.get('stages', [])}
    return approvers


def extend_approvers(initial_stages, stages, merge_to_right=False):
    main_stages = initial_stages
    appending_stages = stages
    if merge_to_right:
        main_stages = stages
        appending_stages = initial_stages

    filtered_appending_stages = []
    for appending_stage in appending_stages:
        if appending_stage in main_stages:
            continue
        all_main_approvers = all_approvers_in_stages(main_stages)
        appending_stage_approvers = all_approvers_in_stages([appending_stage])
        filtered_appending_stage_approvers = appending_stage_approvers - all_main_approvers

        if len(filtered_appending_stage_approvers) > 1:
            filtered_appending_stage = appending_stage
            filtered_appending_stage['stages'] = [
                {'approver': a} for a in sorted(list(filtered_appending_stage_approvers))
            ]
        elif len(filtered_appending_stage_approvers) == 1:
            filtered_appending_stage = {'approver': filtered_appending_stage_approvers.pop()}
        else:
            continue

        filtered_appending_stages.append(filtered_appending_stage)

    initial_stages[:] = main_stages + filtered_appending_stages


def remove_duplicates(stages, keep_firsts=False):
    if not keep_firsts:
        stages[:] = stages[::-1]

    already_in_stages = set()
    res = []

    for stage in stages:

        if 'approver' not in stage:
            res.append(stage)
            continue

        if stage['approver'] in already_in_stages:
            continue

        already_in_stages.add(stage['approver'])
        res.append(stage)

    if not keep_firsts:
        res = res[::-1]

    stages[:] = res


def drop_approvement(stages):
    stages.clear()
    raise FlowDropped()


def _call_table_flow(table_name, **kwargs):
    url = settings.TABLE_FLOW_URL.format(table_name=table_name)
    return requests_retry_session(retries=5).get(url, params=kwargs,
                                                 headers={'Authorization': 'OAuth ' + settings.OK_ROBOT_TOKEN})


def get_persons_by_role_id(role_id):
    return worker.get_users_by_role_id(role_id)


def get_duty_by_service_slug(service_slug):
    return worker.get_on_duty_by_service_slug(service_slug)


def call_table_flow(table_name, **kwargs):
    res = _call_table_flow(table_name, **kwargs)

    if res.status_code > 299:
        raise TableFlowExecutionError(code=res.status_code, table_name=table_name, response=res.text)

    return res.json()
