from logging import getLogger

from django.utils.translation import gettext_lazy as _

from ok.approvements.tracker import collect_issues_for_approvements
from ok.notifications.base import BaseNotification
from ok.notifications.utils import get_message_id
from ok.utils.attrs import get_attribute
from ok.utils.staff import StaffUser

import ok.utils.wiki as wiki


def _get_approvement_message_id(approvement):
    return get_message_id('approvement', approvement.id, approvement.created)


logger = getLogger(__name__)


class ApprovementBaseNotification(BaseNotification):

    subject_prefix = None

    def _get_receiver_logins(self):
        raise NotImplementedError

    def _add_users_to_context(self, context):
        logins = (
            self.initiator,
            self.instance.author,
        )
        users_map = StaffUser.fetch(logins)
        context['initiator'] = users_map.get(self.initiator)
        context['author'] = users_map.get(self.instance.author)

    def get_context(self):
        collect_issues_for_approvements([self.instance])
        context = super().get_context()
        self._add_users_to_context(context)
        context['stages'] = list(self.instance.stages.all())
        context['root_stages'] = list(
            self.instance.stages
            .root()
            .prefetch_related('stages')
        )
        context['formatted_text'] = wiki.format_markup_safe(self.instance.text, default=self.instance.text)
        return context

    def get_receivers(self):
        return ['%s@yandex-team.ru' % l for l in self._get_receiver_logins()]

    def get_subject(self):
        summary = get_attribute(self.instance, 'issue.summary', '')
        return f'[{self.subject_prefix}] ({self.instance.object_id}) {summary}'

    def get_thread_id(self):
        return _get_approvement_message_id(self.instance)


class ApprovementToResponsibleNotification(ApprovementBaseNotification):
    def get_receivers(self):
        receivers = super().get_receivers()
        if self._is_sending_to_author():
            return receivers
        return receivers + self.instance.info_mails_to

    def _is_sending_to_author(self):
        return not bool(self.instance.info_mails_to)

    def _get_receiver_logins(self):
        if self._is_sending_to_author():
            return [self.instance.author]
        return []


class ApprovementExampleNotification(ApprovementBaseNotification):
    """
    Просто пример нотификации для вёрстки писем
    """
    subject_prefix = _('EXAMPLE')
    template_name = 'email/approvements/example.html'

    def send(self, **kwargs):
        # Чтобы случайно не отправить однажды
        raise NotImplementedError


class ApprovementBulkNotificationBase(BaseNotification):

    subject = None

    def __init__(self, initiator=None, **kwargs):
        super().__init__(instance=None, initiator=initiator, **kwargs)

    def get_subject(self):
        return str(self.subject)

    def _get_receiver_logins(self):
        raise NotImplementedError

    def get_receivers(self):
        return ['%s@yandex-team.ru' % l for l in self._get_receiver_logins()]


class ApprovementRequiredNotification(ApprovementBaseNotification):

    subject_prefix = _('Approval action required')
    template_name = 'email/approvements/required.html'

    def _get_receiver_logins(self):
        return {stage.approver for stage in self.kwargs['current_stages']}


class ApprovementApprovedByResponsibleNotification(ApprovementBaseNotification):

    subject_prefix = _('Approved')
    template_name = 'email/approvements/approved_by_responsible.html'

    def _get_receiver_logins(self):
        return self.kwargs['receivers']


class ApprovementFinishedNotification(ApprovementToResponsibleNotification):

    subject_prefix = _('Approval process finished')
    template_name = 'email/approvements/finished.html'


class ApprovementStoppedNotificationBase(ApprovementToResponsibleNotification):
    def _get_receiver_logins(self):
        receivers = {stage.approver for stage in self.kwargs['current_stages']}
        if self._is_sending_to_author():
            receivers.add(self.instance.author)
        receivers.discard(self.initiator)
        return receivers


class ApprovementSuspendedNotification(ApprovementStoppedNotificationBase):

    subject_prefix = _('Approval process suspended')
    template_name = 'email/approvements/suspended.html'


class ApprovementCancelledNotification(ApprovementStoppedNotificationBase):

    subject_prefix = _('Approval process calcelled')
    template_name = 'email/approvements/cancelled.html'


class ApprovementQuestionNotification(ApprovementToResponsibleNotification):

    subject_prefix = _('Question about approval')
    template_name = 'email/approvements/question.html'


class ApprovementReminderNotification(ApprovementBulkNotificationBase):

    subject = _('Approval processes await')
    template_name = 'email/approvements/reminder.html'

    def get_context(self):
        context = super().get_context()
        collect_issues_for_approvements(context['approvements'])
        return context

    def _get_receiver_logins(self):
        return [self.kwargs['receiver']]


class ApprovementOverdueNotification(ApprovementBulkNotificationBase):

    subject = _('Approval awaits too long')
    template_name = 'email/approvements/overdue.html'

    def get_context(self):
        context = super().get_context()
        context['approvement_to_current_stages'] = dict(context['approvement_to_current_stages'])
        collect_issues_for_approvements(context['approvement_to_current_stages'])
        return context

    def get_receivers(self):
        receiver = self.kwargs['receiver']
        if '@' in receiver:
            return [receiver]
        return ['%s@yandex-team.ru' % receiver]
