from typing import Dict

from django.utils import translation

from ok.core.context import get_base_context
from ok.notifications import notify
from ok.utils.staff import get_user_to_language


class BaseNotification(object):

    template_name = None

    def __init__(self, instance, initiator=None, **kwargs):
        self.instance = instance
        self.initiator = initiator
        self.kwargs = kwargs

    def get_subject(self):
        raise NotImplementedError

    def get_receivers(self):
        raise NotImplementedError

    def get_thread_id(self):
        return None

    def get_template_name(self):
        return self.template_name

    def get_context(self):
        context = get_base_context()
        context['instance'] = self.instance
        context['initiator'] = self.initiator
        context.update(self.kwargs)
        return context

    def get_receiver_to_language(self) -> Dict[str, str]:
        login_to_mail = {
            it.split('@')[0]: it
            for it in self.get_receivers()
        }
        login_to_lang = get_user_to_language(login_to_mail)
        return {
            mail: login_to_lang.get(login) or 'en'
            for login, mail in login_to_mail.items()
        }

    def send(self, **kwargs):
        context = self.get_context()
        template_name = self.get_template_name()

        kwargs.setdefault('initiator', self.initiator)
        kwargs.setdefault('message_id', None)
        kwargs.setdefault('in_reply_to', self.get_thread_id())

        if 'reply_to' not in kwargs and self.initiator:
            kwargs['reply_to'] = f'{self.initiator}@yandex-team.ru'

        for receiver, lang in self.get_receiver_to_language().items():
            with translation.override(lang):
                notify(
                    transport='email',
                    template_name=template_name,
                    context=context,
                    subject=self.get_subject(),
                    receiver=receiver,
                    **kwargs
                )

    def __call__(self, **kwargs):
        self.send(**kwargs)
