import json
import re
import waffle

from logging import getLogger
from typing import Dict, List

from cached_property import cached_property
from dateutil import parser
from django.conf import settings
from django.template.loader import render_to_string
from django.utils.translation import ugettext_noop as _
from model_utils import Choices
from startrek_client.exceptions import NotFound
from ylog.context import log_context

from ok.approvements.choices import (
    APPROVEMENT_HISTORY_EVENTS,
    APPROVEMENT_STAGE_APPROVEMENT_SOURCES,
    APPROVEMENT_STATUSES,
)
from ok.approvements.controllers import ApprovementController
from ok.approvements.models import Approvement, create_history_entry
from ok.core.context import get_base_context
from ok.notifications.approvements import ApprovementQuestionNotification
from ok.tracker.base import client, st_connection

from .models import CheckedComment, Queue, Trigger


logger = getLogger(__name__)


OK_TEXTS = {
    'ok', 'okey', 'okay', 'ook', 'ок', 'оок', 'окей', 'окай',
    'окс', 'оки', 'да', 'подтверждаю', 'согласовано',
}
EXCEPT_TEXTS = {'я', 'спасибо'}


class IssueController:

    def __init__(self, key):
        self.key = key

        self.raw_comments = None
        self.comments = None

    @cached_property
    def approvement(self):
        return (
            Approvement.objects
            .filter(object_id=self.key, status=APPROVEMENT_STATUSES.in_progress)
            .first()
        )

    @cached_property
    def approvement_ctl(self):
        return ApprovementController(self.approvement)

    @cached_property
    def stages(self):
        return {s.approver: s for s in self.approvement.stages.active()}

    @cached_property
    def last_checked_comment_id(self):
        return (
            self.approvement.checked_comments
            .order_by('-comment_id')
            .values_list('comment_id', flat=True)
            .first()
        )

    @cached_property
    def issue(self):
        return client.issues[self.key]

    def run(self):
        if not self.approvement:
            return

        with log_context(approvement=self.approvement.id):
            self.raw_comments = self.get_raw_comments()
            self.comments = self.hydrate()
            self.filter_comments()
            self.set_ok_and_save()
            self.update_approvement()
            self.update_ticket()

    def get_raw_comments(self):
        return list(self.issue.comments.get_all())

    def hydrate(self):
        return [
            CheckedComment(
                approvement=self.approvement,
                author=comment.createdBy.id,
                text=comment.text,
                comment_id=comment.id,
                comment_created_at=parser.parse(comment.createdAt),
            )
            for comment in self.raw_comments
        ]

    def filter_comments(self):
        comments = []
        for comment in self.comments:
            need_check = (
                self.comment_is_not_checked(comment)
                and self.comment_was_created_after_approvement(comment)
                and self.author_is_approver(comment)
            )
            if need_check:
                comments.append(comment)
        self.comments = comments

    def comment_was_created_after_approvement(self, comment):
        return comment.comment_created_at > self.approvement.created

    def comment_is_not_checked(self, comment):
        return not self.last_checked_comment_id or comment.comment_id > self.last_checked_comment_id

    def author_is_approver(self, comment):
        return comment.author in self.stages

    def set_ok_and_save(self):
        for comment in self.comments:
            comment.is_ok = self.is_ok(comment.text)
            comment.save()

    @staticmethod
    def is_ok(text):
        rgx = re.compile(r'[,.!-]')
        text = re.sub(rgx, ' ', text.strip().lower())
        words_set = set(text.split()) - EXCEPT_TEXTS
        return bool(words_set) and words_set.issubset(OK_TEXTS)

    def update_approvement(self):
        # Сортируем комменты в обратном порядке этапов согласования,
        # чтобы не спамить отбивками тех, кто уже согласовал
        comments = sorted(self.comments, key=lambda x: -self.stages[x.author].position)
        for comment in comments:
            stage = self.stages[comment.author]
            if comment.is_ok:
                self.approvement_ctl.approve(
                    stages=[stage],
                    initiator=comment.author,
                    approvement_source=APPROVEMENT_STAGE_APPROVEMENT_SOURCES.comment,
                )
                logger.info('Set is_approved for stage %s by comment %s', stage.id, comment.id)
            else:
                create_history_entry(
                    obj=stage,
                    event=APPROVEMENT_HISTORY_EVENTS.question_asked,
                    user=comment.author,
                )
                ApprovementQuestionNotification(
                    instance=self.approvement,
                    initiator=comment.author,
                    comment=comment.text,
                ).send()
                logger.info('Question for stage %s by comment %s', stage.id, comment.id)

    def update_ticket(self):
        have_non_ok_comment = any(not comment.is_ok for comment in self.comments)
        if have_non_ok_comment:
            self.issue.update(approverHasQuestion='yes')


def is_comment_removed(issue, comment_id):
    if not comment_id:
        return True
    try:
        issue.comments[comment_id]
    except NotFound:
        return True
    else:
        return False


def find_comment_remover(issue, comment_id):
    """
    Находит логин пользователя, который удалил коммент c ID = `comment_id` в тикете `issue`.
    Если событие не найдено, отдаёт None
    """
    if not comment_id:
        return None
    for event in issue.changelog.get_all():
        if not event.comments:
            continue
        removed_comments = event.comments.get('removed', [])
        if any(str(c.id) == str(comment_id) for c in removed_comments):
            return event.updatedBy.login


def restore_approvement(issue, approvement, initiator, is_comment_initial=False):
    """
    Восстанавливает коммент с согласованием
    """
    if not waffle.switch_is_active('enable_approvement_restore'):
        return
    context = get_base_context()
    context['instance'] = approvement
    context['initiator'] = initiator
    context['is_comment_initial'] = is_comment_initial
    text = render_to_string('tracker/restore.wiki', context)
    comment = issue.comments.create(text=text, summonees=[] if is_comment_initial else [approvement.author])
    approvement.tracker_comment_short_id = comment.id
    approvement.tracker_comment_id = comment.longId
    approvement.save(update_fields=['tracker_comment_short_id', 'tracker_comment_id', 'modified'])


TRIGGER_CREATE_TYPE = Choices(
    ('complete', _('trigger.type.complete')),
    ('draft', _('trigger.type.draft')),
)


def create_trigger(queue: str, **trigger_params) -> str:
    request = _get_request_for_trigger_create(**trigger_params)
    trigger = st_connection.request(
        method="POST",
        path=f'/v2/queues/{queue}/triggers/',
        data=request,
    )
    queue_model, _ = Queue.objects.get_or_create(name=queue)
    Trigger.objects.create(
        queue=queue_model,
        tracker_id=trigger.id,
    )
    return trigger.self


def _get_request_for_trigger_create(
    name: str,
    trigger_create_type: str,
    groups: List[str],
    is_parallel: bool = False,
    oauth_token: str = None,
    stages: Dict = None,
    flow_name: str = None,
    flow_context: Dict = None,
) -> Dict:
    # for tracker api POST /v2/queues/<QUEUE>/triggers/
    # https://st-api.yandex-team.ru/docs/#operation/updateTriggerApiV2
    flow_context = flow_context or {}
    stages = stages or []
    groups_uri = '&groups='.join(groups)

    if trigger_create_type == TRIGGER_CREATE_TYPE.complete:
        body = {
            "object_id": "{{issue.key}}",
            "uid": "{{currentDateTime}}",
            "text": "smtxt",
            "groups": groups,
            "is_parallel": is_parallel,
        }
        if flow_name:
            body['flow_name'] = flow_name
            body['flow_context'] = {'author_login': '{{issue.author.login}}'}
            context = {k: '{{' + v + '}}' for k, v in flow_context.items()}
            body['flow_context'].update(context)
        else:
            body['stages'] = stages
        create_approvement_action = {
            'type': 'Webhook',
            'id': 1,
            'endpoint': f'{settings.OK_URL}/api/approvements/',
            'method': 'POST',
            'contentType': 'application/json; charset=UTF-8',
            'authContext': {'headerName': 'Authorization',
            'type': 'oauth',
            'accessToken': oauth_token,
            'tokenType': 'OAuth'},
            'body': json.dumps(body, indent=4),
        }
        add_comment_action = {
            'type': 'CreateComment',
            'id': 2,
            'text': (
                f'{{{{=<% %>=}}}}{{{{iframe src="{settings.OK_URL}tracker?_embedded=1&groups={groups_uri}'
                f'&object_id=<%issue.key%>&uid=<%currentDateTime%>" frameborder=0 width=100% height=400px}}}}'
            ),
            'fromRobot': False
        }
        actions = [create_approvement_action, add_comment_action]
    else:
        if flow_name:
            approvers_query = '&flow_name=new_chain_flow&flow_context.author_login=<%issue.author.login%>'
            for k, v in flow_context.items():
                approvers_query += f'&flow_context.{k}=<%{v}%>'
        else:
            approvers_query = ''.join(
                f'&users={stage["approver"]}' for stage in stages
            )
        add_comment_action = {
            'type': 'CreateComment',
            'id': 1,
            'text': (
                f'{{{{=<% %>=}}}}{{{{iframe src="{settings.OK_URL}tracker?_embedded=1&groups={groups_uri}{approvers_query}'
                f'&object_id=<%issue.key%>&uid=<%currentDateTime%>" frameborder=0 width=100% height=400px}}}}'
            ),
            'fromRobot': False
        }
        actions = [add_comment_action]

    return {
        "actions": actions,
        "conditions":[{
            'type': 'CommentFullyMatchCondition',
            'word': 'Согласование',
            'ignoreCase': True,
            'removeMarkup': False,
            'noMatchBefore': False,
        }],
        "name": name,
    }
