import json

from urllib.parse import urljoin

from django.conf import settings
from django.urls import reverse

from ok.tracker.base import client, ISSUE_KEY_RE
from ok.tracker.models import Queue


def get_queue_name(object_id):
    """
    Вычисляет название очереди трекера по object_id
    """
    if not object_id:
        return None
    data = ISSUE_KEY_RE.match(object_id)
    return data.group('queue').upper() if data else None


def get_or_create_queue_by_object_id(object_id):
    queue_name = get_queue_name(object_id)
    if not queue_name:
        return None

    queue, created = Queue.objects.get_or_create(name=queue_name)
    if created:
        from ok.tracker.tasks import check_tracker_queues_fields
        check_tracker_queues_fields.delay(name=queue_name)
    return queue


def get_queue_by_object_id(object_id):
    queue_name = get_queue_name(object_id)
    if not queue_name:
        return None

    return Queue.objects.filter(name=queue_name).first()


def _is_valid_trigger_action(action, valid_enpoint):
    if action['type'] != 'Webhook':
        return False

    try:
        action_body = json.loads(action['body'])
    except ValueError:
        return False

    return (
        action['method'] == 'POST'
        and action['endpoint'] == valid_enpoint
        and 'issue_key' in action_body
        and action_body['issue_key'].replace(' ', '') == '{{issue.key}}'
    )


def _is_valid_trigger_condition(condition):
    return condition['type'] == 'Event.comment-create'


def is_valid_trigger(trigger):
    if not trigger['active']:
        return False

    valid_enpoint = urljoin(settings.OK_URL, reverse('private_api:comments:added_trigger'))
    has_valid_action = any(_is_valid_trigger_action(a, valid_enpoint) for a in trigger['actions'])
    has_valid_condition = any(_is_valid_trigger_condition(a) for a in trigger['conditions'])
    return has_valid_action and has_valid_condition


def check_queue_has_triggers(queue):
    return any(is_valid_trigger(t) for t in client.queues[queue].triggers)


def check_queue_allow_externals(queue):
    return client.queues[queue].allowExternals
