import logging

import waffle
from django.db import transaction
from startrek_client.exceptions import NotFound, Forbidden
from ylog.context import log_context

from ok.celery_app import app
from ok.tracker.controllers import IssueController
from ok.tracker.models import Queue
from ok.tracker.queues import check_queue_has_triggers, check_queue_allow_externals
from ok.approvements.choices import APPROVEMENT_STATUSES
from ok.approvements.models import Approvement

logger = logging.getLogger(__name__)


@app.autoretry_task(max_retries=3)
@transaction.atomic
def update_stage_from_comment(issue_key):
    # Note: Поддерживается параллельный запуск таски на одном issue_key.
    # Успешно отработает таска, которая первой создаст CheckedComment.
    # Последующие таски упадут при попытке создать CheckedComment с тем же comment_id.
    # При перезапуске им останется обработать комменты, созданные после запуска первой таски.
    controller = IssueController(issue_key)
    controller.run()


@app.autoretry_task(max_retries=3)
def check_tracker_queues_fields(**kwargs):
    if not waffle.switch_is_active('enable_queues_check'):
        return

    for queue in Queue.objects.filter(**kwargs):
        with log_context(queue=queue.name):
            try:
                has_triggers = check_queue_has_triggers(queue.name)
                allow_externals = check_queue_allow_externals(queue.name)
            except NotFound:
                logger.warning('Delete unknown tracker queue: %s', queue.name)
                queue.delete()
            except Exception as exc:
                logger.exception('Could not check queue triggers: %s', exc)
            else:
                if queue.has_triggers != has_triggers:
                    queue.has_triggers = has_triggers
                if queue.allow_externals != allow_externals:
                    queue.allow_externals = allow_externals
                queue.save()


@app.task
def update_unconfigured_queue_issue_stages():
    issue_keys = (
        Approvement.objects
        .filter(
            tracker_queue__has_triggers=False,
            status=APPROVEMENT_STATUSES.in_progress,
        )
        .values_list('object_id', flat=True)
    )
    for key in issue_keys:
        try:
            update_stage_from_comment(key)
        except NotFound:
            logger.warning('Issue not found: %s', key)
            continue
        except Forbidden:
            logger.warning('Issue forbidden: %s', key)
        except Exception:
            update_stage_from_comment.delay(key)
