import logging

import requests

from typing import Dict

from django.conf import settings
from django.utils.translation import get_language
from ids.exceptions import BackendError
from ids.registry import registry

from ok.utils.attrs import get_attribute
from ok.utils.http_session import requests_retry_session
from ok.utils.lists import chunks
from ok.utils.tvm import get_service_ticket, get_user_ticket


logger = logging.getLogger(__name__)


def get_staff_repo(resource_type):
    repo = registry.get_repository(
        service='staff',
        resource_type=resource_type,
        user_agent='ok',
        oauth_token=settings.OK_ROBOT_TOKEN,
        timeout=settings.STAFF_API_TIMEOUT,
    )
    return repo


def get_staff_iter(resource_type, params):
    repo = get_staff_repo(resource_type)
    params.setdefault('_limit', 1000)
    params.setdefault('_sort', 'id')
    return repo.getiter(params)


def get_staff_users_iter(params):
    return get_staff_iter(
        resource_type='person',
        params=params,
    )


def get_staff_users_by_logins(login_list, params):
    for login_chunk in chunks(login_list):
        params['login'] = ','.join(login_chunk)
        yield from get_staff_users_iter(params)


def get_staff_groupmembership_iter(params):
    return get_staff_iter(
        resource_type='groupmembership',
        params=params,
    )


def get_staff_group_memberships(group_urls):
    memberships = get_staff_groupmembership_iter({
        'group.url': ','.join(group_urls),
        'person.official.is_dismissed': False,
        '_fields': 'group.url,person.login',
    })
    for membership in memberships:
        yield {
            'group_url': membership['group']['url'],
            'login': membership['person']['login'],
        }


class StaffUser:

    def __init__(self, username, first_name='', last_name='', affiliation=''):
        self._username = username
        self._first_name = first_name
        self._last_name = last_name
        self.affiliation = affiliation

    @property
    def full_name(self):
        return f'{self.first_name} {self.last_name}'

    @property
    def first_name(self):
        lang = get_language()
        return self._first_name.get(lang) or self._first_name.get('en') or ''

    @property
    def last_name(self):
        lang = get_language()
        return self._last_name.get(lang) or self._last_name.get('en') or ''

    @classmethod
    def _build(cls, data):
        return cls(
            username=data['login'],
            first_name=get_attribute(data, 'name.first', {}),
            last_name=get_attribute(data, 'name.last', {}),
            affiliation=get_attribute(data, 'official.affiliation', ''),
        )

    @classmethod
    def fetch(cls, logins):
        """
        Возвращает словарь {login: StaffUser}.
        Если каких-то пользователей получить не удалось,
        они всё равно будут в словаре.
        У них просто не будет ничего кроме логина.
        """
        result = {login: cls(login) for login in logins}
        try:
            users = list(get_staff_users_iter({
                'login': ','.join(logins),
                '_fields': 'login,name.first,name.last,official.affiliation',
            }))
        except BackendError as exc:
            logger.warning('Could not get staff users %s: %s', logins, exc)
        else:
            result.update((i['login'], cls._build(i)) for i in users)
        return result

    def __str__(self):
        return self.username


def get_user_to_language(logins) -> Dict[str, str]:
    users = get_staff_users_iter({
        'login': ','.join(logins),
        '_fields': 'login,language.ui',
    })
    return {
        it['login']: it['language']['ui']
        for it in users
    }


def get_user_language(login) -> str:
    user_to_lang = get_user_to_language([login])
    if user_to_lang:
        return user_to_lang.popitem()[1]


class GapNewhireCountsApiError(Exception):
    pass


def get_gap_newhire_data(request):
    session = requests_retry_session()
    try:
        response = session.get(
            url=f'{settings.STAFF_PROFILE_API}gap_newhire_counts/',
            headers={
                'X-Ya-Service-Ticket': get_service_ticket(settings.TVM_STAFF_CLIENT_ID),
                'X-Ya-User-Ticket': get_user_ticket(request),
            },
            timeout=settings.STAFF_API_TIMEOUT,
        )
    except requests.exceptions.RequestException:
        logger.error('Gap-newhire-counts API is not responding')
        raise GapNewhireCountsApiError('Gap-newhire-counts API is not responding')

    if not response.ok:
        logger.error(
            'Gap-newhire-counts API responded with status %s: %s',
            response.status_code,
            response.content.decode('utf-8'),
        )
        raise GapNewhireCountsApiError(
            f'Gap-newhire-counts API wrong status {response.status_code}',
        )

    return response.json()
