# coding: utf-8
from __future__ import unicode_literals, absolute_import

import json
import itertools

from django import http
from django.apps import apps
from django.db.models.query import QuerySet

try:
    from mptt.models import MPTTModel
except ImportError:
    MPTTModel = None

from emission.core.utils import now_str

from . import forms
from .controller import controller
from ..utils import streaming_multipart_response, chunked_queryset


CONTENT_TYPE_JSON = "application/json; charset=utf-8"


def log_slice(request):
    form = forms.SliceForm(request.GET)

    if form.is_valid():
        start = form.cleaned_data['start']
        stop = form.cleaned_data['stop']

        try:
            data = controller[start:stop]
        except KeyError:
            data = []

        if form.cleaned_data['multipart']:
            return streaming_multipart_response(
                content_iterator=itertools.imap(json.dumps, data),
                content_type=CONTENT_TYPE_JSON,
            )
        else:
            return http.HttpResponse(
                content=json.dumps(list(data)),
                content_type=CONTENT_TYPE_JSON
            )

    return http.HttpResponseBadRequest(form.errors.as_text())


def meta(request):
    form = forms.MetaForm(request.GET)

    if not form.is_valid():
        return http.HttpResponseBadRequest(form.errors.as_text())

    data = {
        'last_message': controller.get_one(controller.get_last_id()),
        'entities': forms.MODELS,
    }

    current_id = form.cleaned_data['current']
    if current_id:
        next_id = controller.get_next_id(current_id)
        if next_id:
            data['next_message'] = controller.get_one(next_id)

    return http.HttpResponse(json.dumps(data),
                             content_type=CONTENT_TYPE_JSON)


def message_generator(queryset):
    for obj in queryset:
        message = {
            'id': 0,
            'action': 'modify',
            'creation_time': now_str(),
            'data': controller.storage.serialize_objects([obj]),
        }
        yield json.dumps(message)


def entities_data(request):
    form = forms.EntityForm(request.GET)

    if form.is_valid():
        app_label, model_name = form.cleaned_data['name'].split('.', 1)

        model = apps.get_model(app_label, model_name)

        qs = model._default_manager.all()
        if isinstance(qs, QuerySet):
            if MPTTModel and issubclass(model, MPTTModel):
                qs = qs.order_by('tree_id', 'lft')
            else:
                qs = qs.order_by('pk')
            qs = chunked_queryset(qs)

        if form.cleaned_data['multipart']:
            return streaming_multipart_response(
                content_iterator=message_generator(qs),
                content_type=CONTENT_TYPE_JSON,
            )
        else:
            message = {
                'id': 0,
                'action': 'modify',
                'creation_time': now_str(),
                'data': controller.storage.serialize_objects(qs)
            }

            return http.HttpResponse(
                content=json.dumps(message),
                content_type=CONTENT_TYPE_JSON,
            )

    return http.HttpResponseBadRequest(form.errors.as_text())
