# coding: utf-8
from __future__ import unicode_literals

import hashlib
import time

from django.apps import apps
from django.db import transaction
from django import http

from emission.core.exceptions import EmissionImproperlyConfigured
from emission.core.utils import import_object


def load_model(name):
    app_name, model_name = name.rsplit('.', 1)

    model = apps.get_model(app_name, model_name)

    if model:
        return model

    return import_object(name)


def isexception(obj):
    return isinstance(obj, type) and issubclass(obj, Exception)


def get_setting(settings, name, default=EmissionImproperlyConfigured):
    try:
        return getattr(settings, name)
    except AttributeError:

        if isexception(default):
            raise default(name)

        elif callable(default):
            return default(name)

        else:
            return default


@transaction.atomic
def chunked_queryset(queryset, limit=None):
    from emission.django.emission_master.settings import EMISSION_DATABASE_CHUNK_SIZE_LIMIT
    
    limit = limit or EMISSION_DATABASE_CHUNK_SIZE_LIMIT
    offset = 0
    while True:
        results = list(queryset[offset:offset+limit].iterator())

        if results:
            for result in results:
                yield result
            offset += limit
        else:
            break


def streaming_multipart_response(content_iterator, content_type):
    boundary = '=' * 6 + hashlib.md5(str(time.time())).hexdigest()[:14]
    response_generator = multipart_generator(
        content_iterator=content_iterator,
        boundary=boundary,
        content_type=content_type,
    )
    response_content_type = 'multipart/mixed; boundary="{}"'.format(boundary)

    if hasattr(http, 'StreamingHttpResponse'):
        return http.StreamingHttpResponse(
            streaming_content=response_generator,
            content_type=response_content_type,
        )
    else:
        return http.HttpResponse(
            content=response_generator,
            content_type=response_content_type,
        )


def multipart_generator(content_iterator, boundary, content_type):
    PART_TEMPLATE = (
        "--{boundary}\r\n"
        "Content-Type: {content_type}\r\n"
        "\r\n"
        "{content}\r\n"
        "\r\n"
    )

    for content in content_iterator:
        yield PART_TEMPLATE.format(
            boundary=boundary,
            content_type=content_type,
            content=content,
        )

    yield "--{}--\r\n".format(boundary)
