# coding: utf-8
from functools import partial
from datetime import datetime
from django.core.management.base import BaseCommand
from django.db.models import F, Q
from django.conf import settings
from libra.books.models import PurchaseRequestOneBook, BookItem, ReadHistory


def department(user):
    depts = user.departments
    if len(depts) > 1:
        if depts[1].id == 2871: # Яндекс.РКУБ
            return depts[2].name
    return "Unknown"


def last_reader(book_item):
    s = book_item.readhistory_set
    hist = s.latest('id') if s.exists() else None
    return hist.reader if hist else None


class Command(BaseCommand):
    help = u""" Отчеты по использованию библиотеки за период N месяцев"""
    args = u"<YYYY.mm.dd>"

    def handle(self, *args, **options):
        since = datetime.strptime(args[0], '%Y.%m.%d')

        for func in (self.purchase, self.returns, self.lost, self.expired):
            self.write_file(partial(func, since), func.__name__ + '.tsv')

    def purchase(self, since):
        requests = PurchaseRequestOneBook.objects.select_related(
            'request', 'request__customer').filter(
                request__created_at__gt=since)

        yield ('department', 'login', 'title', 'cost', 'count', 'created_at')

        for book in requests:
            yield (department(book.request.customer),
                   book.request.customer.login,
                   book.title,
                   book.cost1,
                   book.count,
                   book.request.created_at,
            )

    def returns(self, since):
        history = ReadHistory.objects.select_related('reader', 'book')\
            .filter(get_date__gt=since,
                    return_date__lte=F('get_date') - settings.BOOK_HOLD_PERIOD,
            )

        yield ('department', 'login', 'title', 'get', 'return')

        for record in history:
            yield (department(record.reader),
                   record.reader.login,
                   record.book.title,
                   record.get_date,
                   record.return_date
            )

    def lost(self, since):
        yield ('book', 'title', 'reg_date', 'cost', 'last_reader')

        for book_item in BookItem.objects.select_related('book').filter(
                readhistory__get_date__gt=since,
                location_staff=None,
                shelf__isnull=True
        ):

            reader = last_reader(book_item)

            yield (department(reader),
                   book_item.book,
                   book_item.book.title,
                   book_item.reg_date,
                   book_item.cost,
                   reader.login,
            )

    def expired(self, since):
        history = ReadHistory.objects.select_related('reader', 'book')\
            .filter(get_date__gt=since).filter(
                Q(return_date=None) |
                Q(return_date__gt=F('get_date') - settings.BOOK_HOLD_PERIOD)
            )

        yield ('department', 'login', 'title', 'taken', 'returned')

        for record in history:
            yield (department(record.reader),
                   record.reader.login,
                   record.book.title,
                   record.get_date,
                   record.return_date
            )

    def write_file(self, func, filename):
        with open(filename, 'w') as f:
            print "Writing data to %s ..." % filename
            for info in func():
                f.write((u'\t'.join(map(unicode, info)) + u'\n').encode('utf-8'))
