# coding: utf-8

from math import ceil
from pytils.translit import translify
from string import maketrans
import datetime
import logging
import re
import pytz
from itertools import imap

from django.contrib.auth.decorators import login_required, permission_required
from django.core.exceptions import ObjectDoesNotExist
from django.db.models import Q, Count, Sum
from django.http import Http404, HttpResponse, HttpResponseBadRequest
from django.shortcuts import get_object_or_404, redirect, render
from django.template.loader import render_to_string
from django.conf import settings

from libra.utils import lego_to_html
from libra.books.forms import AddBookForm, FilterCityBookForm
from libra.books.models import (
    Opinion, Book, BookItem, Publisher, ReadHistory, VariantBookInfo,
    PurchaseRequestOneBook, BookQueue, MOSCOW_OFFICES_IDS, SPB_OFFICES_IDS
)
from libra.books.utils import search_staff, fix_typos, create_atpost
from libra.books.vendors import get_vendor
from libra.books.views.tags import make_tags_rubricator
from libra.shortcuts import response_json
from django_intranet_stuff.models import Staff, Office

from libra.tagging.models import Tag, TaggedItem
from ids.registry import registry


@login_required
def list_books(request, current_tag_id=0, current_office_id=0):
    u"""
        Показывает список книг на главной странице,
        отфильтровав по рубрике и офису, если они указаны.
    """
    book_list = Book.objects.select_related().filter(
        bookitem__is_active=True
    ).order_by('title').annotate(count_of_bookitems=Count('bookitem'))

    # показываем только активные офисы, в которых есть книги
    offices = Office.objects.order_by('name').annotate(
        count_of_bookitems=Count('bookitem')
    ).filter(count_of_bookitems__gt=0, intranet_status=1)
    tag = None
    if current_tag_id and int(current_tag_id) > 0:
        current_tag_id = int(current_tag_id)
        tag = get_object_or_404(Tag, id=current_tag_id)
        book_list = TaggedItem.objects.get_by_model(book_list, tag)
    else:
        current_tag_id = 0

    # Всего книг во всех офисах
    book_count = offices.aggregate(
        Sum('count_of_bookitems'))['count_of_bookitems__sum']
    # Количество книг, у которых офис не известен ("Черная дыра")
    books_in_black_hole = book_list.select_related('bookitem').filter(
        bookitem__office__isnull=True).count()

    current_office = None
    if current_office_id and int(current_office_id) > 0:
        current_office_id = int(current_office_id)
        book_list = book_list.filter(bookitem__office__id=current_office_id)
        current_office = get_object_or_404(Office, id=current_office_id)
        current_office = current_office.short_name
    elif current_office_id and int(current_office_id) == 0:
        book_list = book_list.filter(bookitem__office__isnull=True)
        current_office_id = 0
    else:
        current_office_id = None

    books = book_list

    return render(
        request,
        'list-books.html',
        {
            'books': books,
            'book_count': book_count,
            'books_in_black_hole': books_in_black_hole,
            'tag': tag,
            'current_tag_id': current_tag_id,
            'offices': offices,
            'current_office_id': current_office_id,
            'current_office': current_office
        }
    )


@login_required
def list_books_tag(request, current_tag_id):
    u" Показывает список книг, отобранных по рубрике "
    return list_books(request, current_tag_id)


@login_required
def list_books_office(request, current_tag_id, current_office_id):
    u"""
        Показывает список книг, отобранных по рубрике и офису
        (рубрика может быть =0)
    """
    return list_books(request, current_tag_id, current_office_id)


def list_books_json(request):
    u"""
    Возвращает response со списком книг, названия которых содержат строку [q]
    кодированный в json. Количество записей ограничивается
    GET-параметром limit. Используется в форме добавления книги
    (саджест названия).
    """
    q = request.REQUEST.get('q', '')
    if q:
        # ограничение на кол-во записей
        limit = int(request.REQUEST.get('limit', 30))
        books = list(
            Book.objects.filter(title__istartswith=q)
                .select_related('publisher')
                .order_by('title')
                .values('id', 'title', 'authors_string',
                        'publisher__id', 'publisher__name')
        )[:limit]
        return response_json(books)
    else:
        raise Http404


@login_required(redirect_field_name=settings.REDIRECT_FIELD_NAME)
def add_book(request):
    u"""
        Показывает форму добавления книги / обработка формы и добавление книги.
    """

    # Пришли по GET-запросу, значит показываем пустую форму
    if request.method != 'POST':
        book = None
        url = None
        f = AddBookForm()
        if 'req' in request.GET:
            try:
                book = PurchaseRequestOneBook.objects.get(pk=request.GET['req'])
            except PurchaseRequestOneBook.DoesNotExist:
                pass
        if book:
            url = book.url
            vendor = get_vendor(book.url)
            if vendor and vendor.resolve():
                f.fields['title'].initial = vendor.title
                f.fields['isbn'].initial = vendor.isbn
                f.fields['authors_string'].initial = vendor.authors
                f.fields['publisher_name'].initial = vendor.publisher
                f.fields['year'].initial = vendor.year
                f.fields['location_choices'].initial = 'onhands'
                f.fields['location_staff'].initial = book.request.customer.login
                f.fields['cost'].initial = str(book.cost1)
                if vendor.publisher:
                    try:
                        publisher_id = Publisher.objects.get(name=vendor.publisher).id
                        f.fields['publisher_id'].initial = publisher_id
                    except Publisher.DoesNotExist:
                        pass
        
        return render(
            request,
            'add-book-form.html',
            {
                'form': f,
                'showerrors': False,
                'url': url
            }
        )
    else:
        f = AddBookForm(request.POST)
        # Показываем ошибки заполнения формы
        if not f.is_valid():
            return render(
                request,
                'add-book-form.html',
                {
                    'form': f,
                    'showerrors': True
                }
            )

    fcd = f.cleaned_data

    if fcd['rfid']:
        book_item = BookItem.objects.all().filter(rfid=fcd['rfid'])
        if book_item:
            return render(
                request,
                'add-book-form.html',
                {
                    'form': f,
                    'rfid_exists_error': u'Экземпляр книги с таким Rfid '
                                         u'уже существует',
                    'showerrors': True
                }
            )

    # Если введено название издательства, ищем его в базе,
    # и если не находим - создаем новую запись
    if fcd['publisher_name']:
        try:
            if fcd['publisher_id']:
                publisher = get_object_or_404(
                    Publisher, id=fcd['publisher_id']
                )
            else:
                publisher = Publisher.objects.filter(
                    name__iexact=fcd['publisher_name'])[0]
        except:
            publisher = Publisher(name=fcd['publisher_name'])
            publisher.save()
    else:
        publisher = None

    # LIBRA-375
    fcd['title'] = fix_typos(fcd['title'])

    # Ищем в метакнигах по названию, если не находим - создаем
    new_book = False
    try:
        book = Book.objects.get(title__iexact=fcd['title'])
    except ObjectDoesNotExist:
        book = Book(
            title=fcd['title'],
            authors_string=fcd['authors_string'],
            publisher=publisher
        )
        new_book = True
        book.save()

    # Определяем владельца книги (компания, сотрудник, кто именно)
    owner = None
    if fcd['ownership_choices'] == 'PR':
        try:
            owner_login = fcd['owner__login_ld'] or fcd['owner']
            try:
                owner = search_staff(owner_login)
            except Staff.DoesNotExist:
                # TODO: Надо пробрасывать ошибку на форму пользователя
                print 'не нашли сотрудника (владельца)', owner_login
        except:
            pass

    # определяем где или у кого находится книга
    location_staff, office = None, None
    if fcd['location_choices'] == 'lib':
        try:
            office = fcd['office']
        except:
            pass
    else:
        loc_login = fcd['location_staff__login_ld'] or fcd['location_staff']
        try:
            location_staff = search_staff(loc_login)
        except Staff.DoesNotExist:
            # TODO: Надо пробрасывать ошибку на форму пользователя
            print 'не нашли сотрудника', loc_login

    # Вычистим ISBN от нецифр
    if fcd['isbn']:
        clean_isbn = ''
        for sym in fcd['isbn']:
            if sym.isdigit():
                clean_isbn += sym
        if clean_isbn[:13]:
            # обрезаем до 13 цифр и приводим к INT
            # (а по хорошему нужно сообщить пользователю об ошибке!)
            # TODO FIXME
            fcd['isbn'] = int(clean_isbn[:13])
        else:
            fcd['isbn'] = None
    else:
        fcd['isbn'] = None

    # ссылается на авторизованного пользователя (модель Staff)
    user = request.user.get_profile()

    bi = BookItem(
        book=book,
        ownership_choices=fcd['ownership_choices'],
        owner=owner,
        office=office,
        location_staff=location_staff,
        is_tablebook=fcd['is_tablebook'],
        is_paper=True,
        comment=fcd['comment'],
        cost=fcd['cost'],
        is_active=True,
        isbn=fcd['isbn'],
        registrator_staff=user,
        year=fcd['year'],
        rfid=fcd['rfid'],
        case=fcd['case'],
        shelf=fcd['shelf'],
    )
    bi.save()
    if location_staff:
        bi.move_to_user(location_staff)

    if bi.current_reader:  # Если книга сейчас на руках
        bi.office = bi.current_reader.office
    elif bi.owner:    # Если книга у владельца и владелец указан
        bi.office = bi.owner.office
    bi.save()

    if new_book:                # если книга новая, то
        create_atpost(bi)       # создаем запись о ней в этушке

    return redirect('view_book', book.id)


@login_required
def add_my_book(request):
    u" Шоткат для сценария 'Добавить свою книгу' "
    return add_book(request, my=True)


@login_required
def download_book(request, book_id):
    u" Скачать электронную версию книги "

    book = get_object_or_404(Book, id=book_id)

    if book.mulca_file:
        # response = HttpResponse(book.mulca_file, mimetype="text/plain")
        response = HttpResponse(book.mulca_file, mimetype="application/pdf")
        # response['Content-Type'] = 'application/pdf';
        # response['Content-Type'] = 'text/plain'
        title = translify(book.title)
        table = maketrans(' ', '_')
        title = title.translate(table)
        r1 = re.compile(r"[^a-zA-Z0-9\_]")
        r2 = re.compile(r"_+")
        title = r1.sub("", title)
        title = r2.sub("_", title)
        title = title + '.pdf'
        response['Content-Disposition'] = 'attachment; filename=' + title
        return response

    return redirect('view_book', book_id)


@login_required
def view_book(request, book_id):
    u" Форма просмотра карточки книги "

    book = get_object_or_404(Book, id=book_id)
    data = {'book': book}

    # Список экземпляров книг
    # TODO: Исключить еще личные книги уволенных
    book_items = book.get_items()
    rearranged_book_items = []

    # попробуем поднять экземпляр у текущего пользователя
    # на первое место (переранжирование)
    if request.user.is_authenticated():
        current_user_id = request.user.get_profile().id
    else:
        current_user_id = None

    for bi in book_items:
        if bi.location_staff and bi.location_staff.id == current_user_id:
            rearranged_book_items.insert(0, bi)
        else:
            rearranged_book_items.append(bi)

    data['book_items'] = rearranged_book_items
    data['book_item_count'] = len(rearranged_book_items)

    # Список годов издания берем из экземпляров книг
    years_list = book_items.values_list(
        'year', flat=True).order_by('year').distinct()
    #print years_list
    if years_list and years_list[0] != None:
        years_list = map(str, years_list)
        data['years'] = ', '.join(years_list)
    else:
        data['years'] = ' - '

    # Определим список рубрик, которым принадлежит книга
    data['tags'] = Tag.objects.get_for_object(book)

    # Получим список книг, пересекающихся по каждой рубрике с текущей
    relevant_books = TaggedItem.objects.get_union_by_model(
        Book.objects.exclude(id=book.id), data['tags'])
    data['relevant_books'] = relevant_books

    if request.user.is_authenticated():
        readhistory = ReadHistory.objects.filter(
            book=book, reader=request.user.get_profile(), return_date__isnull=False)
        opinion = Opinion.objects.filter(
            book=book, reader=request.user.get_profile())
        if readhistory:
            readhistory = readhistory[0]
        if opinion:
            opinion = opinion[0]
    else:
        readhistory, opinion = None, None
    data['readhistory'] = readhistory
    data['opinion'] = opinion

    # Сформируем список рубрик, который будет показан в селекторе рубрик
    # Список всех рубрик вообще
    tags_all = make_tags_rubricator()['all_tags_plain']
    tags_count = len(tags_all)

    if tags_count == 0:  # особый случай!
        tags = None
    elif tags_count <= 10:  # 1 колонка из не более 10 рубрик
        tags = tags_all
    elif tags_count > 10 and tags_count < 16:  # 2 колонки из 11-15 рубрик
        part_index = int(tags_count / 2)
        # поделили на 2 части
        tags = [tags_all[:part_index], tags_all[part_index:]]
    else:   # 4 колонки из >=16 рубрик
        part_index = int(ceil(tags_count / 4.0))
        tags = [
            tags_all[:part_index],
            tags_all[part_index:part_index * 2],
            tags_all[part_index * 2:part_index * 3],
            tags_all[part_index * 3:]
        ]

    data['texts'] = {
        'we_have_your_opinion': u'Об этой книге есть ваш отзыв',
        'you_have_read_give_your_opinion': u'Снять отметку о прочтении',
        'wow_i_have_read_this_book': u'Пометить как прочитанную'
    }

    info_variant = VariantBookInfo.objects.filter(book=book, current=True)
    # Для книг, у которых найден только 1 аналог, сразу выставляем этот вариант
    if len(info_variant) == 1:
        data['cover_image'] = info_variant[0].image_l_url
    else:
        data['cover_image'] = None

    comments = data['comments'] = {}
    if book.at_id:
        comments['add_url'] = (settings.AT_CLUB_URL.format(item_no=book.at_id)
                               + '&for_reply=text')

    # логика добавления в очередь
    if request.user.staff.office_id in MOSCOW_OFFICES_IDS:
        office_id = MOSCOW_OFFICES_IDS[0]
    elif request.user.staff.office_id in SPB_OFFICES_IDS:
        office_id = SPB_OFFICES_IDS[0]
    else:
        office_id = request.user.staff.office_id

    free_books = book.get_items().filter(office_id=office_id,
                                         location_staff=None,
                                         is_active=True).select_related('office')

    free_books = filter(lambda x: not x.is_lost, free_books)

    if (free_books or
        # или человек уже и так взял книгу почитать
        book.get_items().filter(location_staff=request.user.staff).exists()):
        add_to_queue = None
    else:
        # Проверим что пользователь уже не стоит в очереди
        # или не взял уже книгу
        if BookQueue.objects.filter(book=book, user=request.user.staff,
                                    finished=False).exists():
            add_to_queue = 'unsubscribe'
        else:
            add_to_queue = 'subscribe'

    return render(
        request,
        'view-book-form.html',
        {
            'data': data,
            'tags': tags,
            'add_to_queue': add_to_queue,
        }
    )


@login_required
def at_reviews(request, book_id):
    book = get_object_or_404(Book, id=book_id)
    rep = registry.get_repository('at', 'comment')
    utc = pytz.timezone('UTC')
    moscow = pytz.timezone('Europe/Moscow')

    def prepare(comment):
        comment['author'] = get_object_or_404(Staff, uid=comment['author_uid'])

        pub = utc.localize(
            datetime.datetime.strptime(comment['published'], "%Y-%m-%dT%H:%M:%SZ"))
        comment['published'] = pub.astimezone(moscow)

        comment['id'] = int(comment['id'].split('/')[-1])
        comment['parent'] = (int(comment['parent'].split('.')[-1])
                             if comment['parent'] is not None else None)
        comment['link'] = (settings.AT_CLUB_URL + "&parent_id={parent}").format(
            parent=comment['id'], item_no=book.at_id)
        return comment

    if not book.at_id:
        return render(request, 'at-comments.html', [])

    # присваиваем комментариям уровень вложенности
    comments = []
    parents_stack = []
    sub_comments = []
    for comment in imap(prepare, rep.getiter({'uid': settings.AT_CLUB_ID,
                                              'post_no': book.at_id})):
        id_ = comment['id']
        parent = comment['parent']

        if parent is not None:
            for _ in xrange(len(parents_stack)):
                prev = parents_stack[-1]
                if prev == parent:
                    parents_stack.append(id_)
                    level = len(parents_stack) - 1
                    comment['level'] = level if level < 4 else 4
                    break
                else:
                    parents_stack.pop()
            sub_comments.append(comment)
        else:
            parents_stack = [id_]
            comment['level'] = 0
            sub_comments = []
            comments.append((comment, sub_comments))

    return render(request, 'at-comments.html', {'comments': comments})


@login_required(redirect_field_name=settings.REDIRECT_FIELD_NAME)
def edit_book(request, book_id):
    u"""
        Форма редактирования книги (показ/обработка)
        #TODO: Написать функцию
    """
    book = get_object_or_404(Book, id=book_id)
    f = AddBookForm(instance=book)
    return render(
        request,
        'add-book-form.html',
        {
            'form': f
        }
    )


@login_required(redirect_field_name=settings.REDIRECT_FIELD_NAME)
def delete_book(request, book_id):
    u"""
        Операция удаления книги
        #TODO: Написать функцию
    """
    raise Http404


@login_required
def iread_book(request, book_id):
    u"""
        Помечает книгу как прочитанную текущим пользователем
        и сохраняет текст отзыва (если есть)
        (обвязана тестами) (ajax)
    """
    if request.method == 'POST' and request.user.is_authenticated():
        read = request.POST.get('read', '1')
        book = get_object_or_404(Book, pk=book_id)

        if read == '1':
            history = ReadHistory.objects.filter(book=book,
                                                 reader=request.user.get_profile(),
                                                 return_date__isnull=False).order_by('-return_date')
            if history.count() == 0:
                return_date = datetime.datetime.now()
                get_date = return_date - datetime.timedelta(seconds=1)
                readhistory = ReadHistory(book=book, 
                                          reader=request.user.get_profile(),
                                          get_date=get_date,
                                          return_date=return_date)
                readhistory.save()

            opinion_text = request.POST.get('opinion_text', None)
            if opinion_text:  # пришел еще текст отзыва #
                opinion_text = opinion_text.strip()
                if len(opinion_text) > 0:
                    opinion, created = Opinion.objects.get_or_create(
                        book=book,
                        reader=request.user.get_profile(),
                        defaults={'text': opinion_text}
                    )
                    if not created:
                        opinion.text = opinion_text
                        opinion.save()
        # POST-запрос с read=False ->
        # нужно удалить метку "читал эту книгу" и отзыв.
        else:
            ReadHistory.objects.filter(book=book,
                                       reader=request.user.get_profile(),
                                       return_date__isnull=False).delete()
            try:
                opinion = Opinion.objects.get(
                    book=book, reader=request.user.get_profile())
                opinion.delete()
            except:
                pass

        return HttpResponse("saved", "text/plain")
    else:
        raise Http404


@login_required
def user_books(request, user_name=None):
    u"""
    Показывает книги, принадлежащие сотруднику и книги, которые у него на руках
    """
    user = None
    if request.user.is_authenticated():
        if request.user.get_profile() and (
                request.user.get_profile().login_ld == user_name
                or not user_name
            ):
            user = request.user.get_profile()

    if not user:
        user = get_object_or_404(Staff, login_ld=user_name)

    if user:
        user_genitive = user.inflections.genitive

        # личные книги пользователя
        owned_books = BookItem.active.filter(
            owner=user).select_related().order_by('book__title')
        # книги кмпании и коллег, которые у пользователя на руках
        onhand_books = BookItem.active.filter(location_staff=user).filter(
            ~Q(owner=user)).select_related().order_by(
                'ownership_choices', 'book__title')

        read_books_raw = Book.objects.filter(
            readhistory__reader=user,
            readhistory__return_date__isnull=False
        ).order_by('readhistory__return_date', 'title')

        read_books = set()
        for book in read_books_raw:
            if not book in  read_books:
                read_books.add(book)

        return render(
            request,
            'user-books.html',
            {
                'owned_books': owned_books,
                'onhand_books': onhand_books,
                'read_books': read_books,
                'current_user': user,
                'user_genitive': user_genitive,
                'settings': settings
            }
        )
    else:
        logging.warning('LIBRA: Unknown user %s / libra.books.user_books' %
            request.user)
        raise Http404


@login_required
def remove_tag(request):
    u" Удаляет книгу из рубрики (ajax) "
    if request.user.is_authenticated():
        book_id = request.POST.get('book_id', None)
        tag_id = request.POST.get('tag_id', None)

        book = get_object_or_404(Book, id=book_id)
        removing_tag = get_object_or_404(Tag, id=tag_id)

        book_tags = list(book.tags)       # список тегов у книги до удаления
        book.tags = None                  # Удаляем все теги
        for tag in book_tags:        # Снова добавим все теги, кроме удаляемого
            if tag != removing_tag:
                Tag.objects.add_tag(book, '"' + tag.name + '"')

        result = {'success': 1, 'tag_id': tag_id}
    else:
        result = {
            'success': 0,
            'msg': 'Авторизуйтесь, пожалуйста.',
            'tag_id': 0
        }

    return response_json(result)


@login_required
def add_tag(request):
    u" Добавляет книгу в заданную рубрику (ajax) "
    if request.user.is_authenticated():
        book_id = request.POST.get('book_id', None)
        tag_id = request.POST.get('tag_id', None)

        book = get_object_or_404(Book, id=book_id)
        new_tag = get_object_or_404(Tag, id=tag_id)

        book_tags = list(book.tags)       # список тегов у книги до удаления
        if not (new_tag in book_tags):
            Tag.objects.add_tag(book, '"' + new_tag.name + '"')
            result = {'success': 1, 'tag_id': tag_id, 'tag_name': new_tag.name}
        else:
            result = {
                'success': 0,
                'tag_id': tag_id,
                'msg': 'книга уже в этой рубрике'
            }

    else:
        result = {
            'success': 0,
            'msg': 'Авторизуйтесь, пожалуйста.',
            'tag_id': 0
        }

    return response_json(result)


@login_required
def alphabet_list(request):
    u" Выводит список всех книг, сгруппировано по алфавиту "
    from libra.books.utils import group_books

    f = FilterCityBookForm(request.GET)
    books = Book.objects.filter(
        bookitem__is_active=True).order_by('title').distinct()

    if f.is_valid():
        office = f.cleaned_data['office']
        if office is not None:
            books = books.filter(bookitem__office=office)

    # группируем список книг по первой букве/цифре
    grouped_books = group_books(books)

    return render(
        request,
        'alphabet-books-list.html',
        {
            'grouped_books': grouped_books,
            'filter_form': f,
        }
    )


@login_required
def view_lenta(request):
    u" Плоский список новых книг (добавленных и заказанных) "
    bottom_date = datetime.date.today() - datetime.timedelta(days=30)

    # список добавленных книг (на основе экземпляров)
    new_books = BookItem.objects.select_related('book', 'office')\
        .filter(is_active=True, reg_date__gte=bottom_date)\
        .distinct().order_by('-reg_date', 'book__title')\
        .values('book__title', 'reg_date', 'book_id', )

    # список заказанных книг
    ordered_books = PurchaseRequestOneBook.objects.select_related('request')\
        .filter(request__created_at__gte=bottom_date)\
        .order_by('-request__modified_at')\
        .values('title', 'request__created_at', 'request__id')

    books_dict, books = {}, []

    def explode_book_list(book_list, dt, title, uniq_id, type):
        u"""
        Добавляет в dict книги из выборки, приводя поля к единым наименованиям
        """
        uni = {}
        # Удалим дубликаты и разобъем плоский список на группы по дате
        for b in book_list:
            d = b[dt].date()
            if d not in books_dict:
                books_dict[d] = []
            if b[uniq_id] not in uni:
                books_dict[d].append({
                    'title': b[title],
                    'uniq_id': b[uniq_id],
                    'type': type,
                })
                uni[b[uniq_id]] = 1

    explode_book_list(new_books, 'reg_date', 'book__title', 'book_id', 'new')
    explode_book_list(
        ordered_books, 'request__created_at', 'title', 'request__id', 'order')

    # теперь развернем словарь с книгами в список и упорядочим по дате
    for dt, l in books_dict.items():
        for v in l:
            v.update({'dt': dt})
            books.append(v)

    return render(
        request,
        'books-lenta.html',
        {
            'books': books
        }
    )


@login_required
def igot_book(request):
    u" Изменяет текущее местонахождение экземпляра книги "

    if (not request.method == 'POST'
        or not request.user.is_authenticated()
        or not request.user.get_profile()):
        logging.warning(
            'LIBRA: Posibbly profile of user "%s"'
            ' not found or method!=post / books.views.books.igot_books()' %
                request.user)
        return HttpResponseBadRequest(u'Не найден профиль пользователя')

    bookitem_id = request.POST.get('bookitem_id', None)
    action = request.POST.get('action', None)

    if not bookitem_id or not action:
        raise Http404

    bookitem = get_object_or_404(BookItem, id=bookitem_id)

    owner = bookitem.owner
    user = request.user.get_profile()

    if action == 'get':
        # TODO: запомнить у кого эта книга была до операции
        # TODO: Проверить наличие истории чтения, и если нету -
        # добавить запись.
        bookitem.move_to_user(user)
    elif action == 'return':
        # первый кейс: возврат владельцу
        if owner:
            if user == owner:  # владелец, говорит что кому-то отдал книгу
                pass  # TODO: решить этот кейс через интерфейс
            else:             # пользователь возвращает книгу владельцу
                bookitem.move_to_user(owner)
        else:
            bookitem.move_to_library(user.office, user)
        #TODO: рассмотреть другие варианты передачи,
        # возврата (другому человеку)

    new_link = lego_to_html(render_to_string('components/i-got-book-link.html', {
        'user': request.user, 'bi': bookitem
    }))
    new_location = lego_to_html(render_to_string('components/book-location-text.html', {
        'bi': bookitem
    }))
    result = {
        'success': 1,
        'bookitem_id': bookitem_id,
        'new_link': new_link,
        'new_location': new_location
    }
    return response_json(result)


@login_required
def get_items_inventory(request, book_id):
    bookitems = BookItem.objects.filter(book__id=book_id)
    book = get_object_or_404(Book, id=book_id)
    return response_json({
        'html': render_to_string('inventory/book-items.html', {
            'bookitems': bookitems
        }),
        'book_link': book.get_absolute_url(),
    })


@login_required
def upload_book_file(request):
    pass


def autofill_book(request):
    shop_url = request.POST.get('url', None)
    vendor = get_vendor(shop_url)
    if vendor and vendor.resolve():
        publisher_id = None
        if vendor.publisher:
            try:
                publisher_id = Publisher.objects.get(name=vendor.publisher).id
            except Publisher.DoesNotExist:
                pass

        return response_json({
            'isbn': vendor.isbn,
            'title': vendor.title,
            'authors': vendor.authors,
            'publisher': {'name': vendor.publisher, 'id': publisher_id},
            'year': vendor.year,
        })
    raise Http404


@permission_required('books.change_book')
def no_cover(request):
    page = int(request.GET.get('page', '1'))
    books_per_page = 10
    start = (page - 1) * books_per_page
    end = start + books_per_page
    books = Book.objects.filter(variantbookinfo__current=False).distinct().order_by('id')
    my_books = books[start:end]
    for b in my_books:
        try:
            b.current_vbi = VariantBookInfo.objects.get(book_id=b.id, current=True)
        except VariantBookInfo.DoesNotExist:
            b.current_vbi = None
    return render(
        request,
        'no-cover-books.html',
        {
            'pages': {
                'current': page,
                'total': books.count()
            },
            'books': my_books
        })

@permission_required('books.change_book')
def set_cover(request):
    try:
        vbi = VariantBookInfo.objects.get(pk=request.REQUEST['cover_id'])
        for info in VariantBookInfo.objects.filter(book_id=vbi.book_id):
            info.current = False
            info.save()
        vbi.book.main_cover = None
        vbi.book.save()
        vbi.current = True
        vbi.save()
        return response_json({'result': 'ok'})
    except:
        traceback.print_exc()
        return response_json({'result': 'error'})


