# -*- coding: utf-8 -*-

from django import forms
from django.contrib.auth.decorators import login_required
from django.shortcuts import render
from django.template.loader import render_to_string

from django_intranet_stuff.models import City
from libra.books.models import BookItem, ContainOfficeId
from libra.rfid import Rfid
from libra.shortcuts import response_json


class CaseForm(forms.Form):
    city = forms.ModelChoiceField(queryset=City.objects.all(), label=u'Город')
    case = forms.IntegerField(max_value=100, label=u'Шкаф')
    shelf = forms.IntegerField(max_value=100, label=u'Полка')


@login_required
def choose_case(request):
    u" Назначить книгам полки и шкафы "
    f = CaseForm(request.GET or None)

    if f.is_valid():
        case_value = f.cleaned_data['case']
        shelf_value = f.cleaned_data['shelf']
        city = f.cleaned_data['city']
        office_ids = ContainOfficeId(city.id)
        books = BookItem.active.filter(
            case=case_value,
            shelf=shelf_value,
            location_staff__isnull=True,
            office_id__in=office_ids,
        ).select_related('book')

        return render(
            request,
            'case-form-choose.html',
            {
                'form': f,
                'case_value': case_value,
                'shelf_value': shelf_value,
                'book_items': books,
                'book_item_count': len(books),
                'city': city,
            }
        )

    return render(
        request,
        'case-form.html',
        {
            'form': f
        }
    )


def choose_book_save(request):
    u' Записать шкаф и полку для выбранных экземпляров книг '

    result = {
        'result': 'ok'
    }
    case = request.GET.get('case', None)
    shelf = request.GET.get('shelf', None)
    bookitem_id = request.GET.get('bookitem_id', None)
    booklist_id = bookitem_id.split(",")
    for bi_id in booklist_id:
        try:
            bookitem = BookItem.objects.get(id=bi_id)
            bookitem.case = case
            bookitem.shelf = shelf
            bookitem.save()
        except:
            pass
    return response_json(result)


def choose_book_rfid(request):
    u' Возвращает объект по заданному RFID (экземпляр книги) '

    rfid_string = request.GET.get('rfid', None)
    result = {
        'result': 'ok',
    }
    if rfid_string:
        # мы всегда ожидаем тут ввод в HEX от ридера
        rfid_number = Rfid.hex2decimal(rfid_string)
        if rfid_number:
            result['rfid_number'] = rfid_number
            try:
                bi = BookItem.objects.get(rfid=rfid_number)
                if bi.book.main_cover:
                    book_cover_url = bi.book.main_cover.img_large.url
                else:
                    book_cover_url = None
                result.update({
                    'type': 'book',
                    'bookitem_id': bi.id,
                    'book_cover_url': book_cover_url,
                    'book_title': bi.book.title,
                    'book_authors': bi.book.authors_string,
                    'html': render_to_string('case-book-item.html', {'bi': bi})
                })
            except BookItem.DoesNotExist:
                result.update({
                    'result': 'error',
                    'message': u'Вы поднесли что-то неизвестное нам,'
                               u' проверьте раскладку клавиатуры'
                })
        else:
            result.update({'result': 'error'})

    else:
        result.update({'result': 'error'})

    return response_json(result)
