# coding: utf-8
import random

import feedparser
from datetime import datetime
from pytz import timezone as tz

from django.shortcuts import render_to_response
from django.template import RequestContext
from django.template.loader import render_to_string
from django.conf import settings
from django.contrib.auth.decorators import login_required

from libra.books.models import Book
from libra.books.views.tags import make_tags_columns, make_plain_tags_list
from libra.shortcuts import response_json
from libra.tagging.models import Tag


@login_required(redirect_field_name=settings.REDIRECT_FIELD_NAME)
def index_page(request):
    u'Морда сервиса'

    books_with_cover_10 = Book.objects.filter(
        main_cover__isnull=False,
        bookitem__is_active=True,
        bookitem__ownership_choices='YA'
    ).distinct().order_by('-reg_date')[:15]

    if books_with_cover_10.count() >= 3:
        books_with_cover = random.sample(books_with_cover_10, 3)
    else:
        books_with_cover = []

    for book in books_with_cover:
        book.cover_url = book.main_cover.img_large.url

    exclude_books = []
    front_books = []

    for i in range(3):
        try:
            book_i = books_with_cover[i]
            front_books.append(book_i)
            exclude_books.append(book_i.id)
        except:
            exclude_books.append(None)

    # Список последних добавленных книг,
    # исключая те, который показаны обложками
    last_books = Book.objects.exclude(id__in=exclude_books).filter(
        bookitem__ownership_choices='YA'
    ).distinct().order_by('-reg_date', 'title')[:5]
    # TODO: Фильтровать не уникальные книги (а такое возможно??)

    tags_groups = make_tags_columns()  # 15

    cat_tags_ids = [ct.id for ct in make_plain_tags_list(tags_groups)]
    orphaned_tags = Tag.objects.exclude(id__in=cat_tags_ids)

    return render_to_response('index.html', {
        'book1': front_books[0],
        'book2': front_books[1],
        'book3': front_books[2],
        'more_books': last_books,
        'tags_groups': tags_groups,
        'orphaned_tags': orphaned_tags,
        'is_index_page': True,
    }, context_instance=RequestContext(request))


def service_blog_rss(request):
    '''
    Выводит новости сервиса как часть html-морды,
    которые берет из Этушки (URL ленты хранится в settings.py проекта)
    '''

    result = {'error': False, 'block': '', 'message': ''}
    try:
        if settings.SERVICE_BLOG_RSS_URL:
            feed = feedparser.parse(settings.SERVICE_BLOG_RSS_URL)
            # возьмем только 4 первые записи
            # (этушка самые свежие отдает первыми)
            news = feed.entries[:4]
            if news:
                moscow_tz = tz(settings.TIME_ZONE)
                utc_tz = tz('GMT')

                for n in news:
                    d = n.updated_parsed
                    # форматируем дату из 9-тапла в строку
                    # (дата приходит в GMT поясе, не Москва)
                    server_time = datetime(d[0], d[1], d[2], d[3], d[4], d[5])
                    # Горите в аду таймзоны!
                    # Замена таймзоны на None нужно для фильтра human_datetime
                    n.norm_date = moscow_tz.normalize(
                        utc_tz.localize(server_time)).replace(tzinfo=None)

                block = render_to_string('components/service-blog-rss.html', {
                    'news': news
                })
                result.update(block=block)
            else:
                result.update(error=True, message='No news from %s found!' %
                    (settings.SERVICE_BLOG_RSS_URL, ))

    # Ловим всё что можно и отправляем текст ошибки на клиент,
    # ибо, как показывает практика, RSS нередко отваливается.
    except Exception as e:
        result.update(block='', error=True, message=str(e))

    return response_json(result)
