# coding: utf-8

import copy

from django.contrib.auth.decorators import login_required
from django.shortcuts import get_object_or_404, render

from libra.books.models import BookItem, RfidFormField
from libra.books.views.books import get_items_inventory
from libra.rfid import Rfid


@login_required
def inventory_form(request):
    return render(
        request,
        'inventory/basic-form.html',
    )


def rfid_to_bookitems(rfids):
    u'''
    из текстовых строк с предполагаемыми rfid-кодами
    формирует список экземпляров книг
    '''
    lines = rfids.strip().split('\r\n')
    items = []
    for i in lines:
        rf = Rfid.smart_float2decimal(i)
        if rf:
            bookitems = BookItem.objects.filter(rfid=rf)
            if len(bookitems) > 0:
                for bi in bookitems:
                    items.append((i, rf, bi))
            else:
                # не нашли соотв для введенного rfid-кода
                items.append((i, rf, None))
    return items


@login_required
def erase_rfid(request):
    if request.method == 'POST':
        step = request.GET.get('step', None)
        rfids = request.POST.get('rfids', None)
        items = []
        if rfids:
            items = rfid_to_bookitems(rfids)

        if step == 'select':
            return render(
                request,
                'inventory/erase-rfid-step-2.html',
                {'items': items, 'rfids': rfids},
            )
        elif step == 'do':
            for i in filter(lambda i: bool(i[2]), items):
                print 'deleting ', i[2]
                i[2].delete()
            return render(
                request,
                'inventory/erase-rfid-step-final.html',
                {'items': items},
            )
    else:
        return render(
            request,
            'inventory/erase-rfid.html',
        )


@login_required
def list_inventory(request):
    pass


@login_required
def update_rfid(request):

    bookitem_id = request.POST.get('bookitem_id', None)
    new_rfid = request.POST.get('new_rfid', None)

    book_item = get_object_or_404(BookItem, id=bookitem_id)

    if new_rfid:
        # Прогоним входящий номер RFID
        # через валидатор-нормализатор поля RfidFormField
        rfid_form = RfidFormField()
        rfid = rfid_form.clean(new_rfid)

        book_item.rfid = rfid
    else:
        book_item.rfid = None
    book_item.save()

    book_id = book_item.book.id

    # вернем обновленный фрагмент страницы
    return get_items_inventory(request, book_id)


@login_required
def clone_bookitem(request):
    u'''
    Клонирует экземпляр книги
    и возвращает отрендеренный новый список экземпляров книги
    '''
    bookitem_id = request.POST.get('bookitem_id', None)
    book_item = get_object_or_404(BookItem, id=bookitem_id)

    new_bookitem = copy.deepcopy(book_item)
    new_bookitem.id = None  # сбросим ID, чтобы ORM создал новую запись
    new_bookitem.rfid = None
    new_bookitem.registrator_staff = request.user.get_profile()
    new_bookitem.is_active = True
    new_bookitem.src_choices = 3
    new_bookitem.save()

    new_bookitem.move_to_user(new_bookitem.current_reader)

    # вернем обновленный фрагмент страницы
    return get_items_inventory(request, new_bookitem.book.id)
