# coding: utf-8

import math, traceback

from django.conf import settings
from django.contrib.auth.decorators import login_required, permission_required
from django.core.urlresolvers import reverse
from django.core.mail import EmailMessage
from django.forms.formsets import formset_factory
from django.http import HttpResponseRedirect, HttpResponsePermanentRedirect
from django.shortcuts import get_object_or_404, render
from django.template.loader import render_to_string


from libra.books.forms.orders import RequestForm, RequestOneBookForm
from libra.books.MailManager import MailManager
from libra.books.models import (
    PurchaseRequest, RequestFlow, PurchaseRequestOneBook, MOSCOW_OFFICES_IDS,
    SPB_OFFICES_IDS)
from libra.books.utils import get_boss
from libra.shortcuts import response_json
from django_intranet_stuff.models import Staff

book_form_initial = {'count': 1}

@login_required
def add_request(request):
    '''
    Редирект на новую форму заказа книг
    '''
    add_request_book_form_url = 'https://wiki.yandex-team.ru/users/armineakh/bookpurchases/'
    return HttpResponsePermanentRedirect(add_request_book_form_url)


@login_required
def edit_request(request, request_id=None):
    pass


@login_required
def view_request(request, request_id):
    u'Просмотр заявки на покупку книг'
    purchase_request = get_object_or_404(PurchaseRequest, id=request_id)

    requested_books = purchase_request.requested_books.all()

    return render(
        request,
        'orders/view-request.html',
        {
            'requested_books': requested_books,
            'req': purchase_request
        },
    )


@login_required
def mail_request(request, request_id):
    '''
    вьюха для разработки/отладки сообщения,
    которые отправляется по email при заказе книг
    '''
    purchase_request = get_object_or_404(PurchaseRequest, id=request_id)

    requested_books = purchase_request.requested_books.all()

    return render(
        request,
        'orders/mail.html',
        {
            'requested_books': requested_books,
            'req': purchase_request
        },
    )


@login_required
def approve_request(request, request_id=None, decision='no'):
    pass


@login_required
def list_requests(request):
    requests = (
        PurchaseRequest.objects.all()
            .order_by('-id')
            .select_related()[:100]
    )

    for r in requests:
        r.books_count = r.requested_books.all().count()

    return render(
        request,
        'orders/list-requests.html',
        {
            'requests': requests
        },
    )


@login_required
def manage_requests(request):
    page = int(request.REQUEST.get('page', '1'))
    per_page = 50
    start = (page - 1) * per_page
    end = page * per_page
    total_count = PurchaseRequest.objects.count()

    requests = PurchaseRequest.objects.order_by('-id').select_related()[start:end]
    for r in requests:
        books_count = 0
        price = 0
        for book in r.requested_books.all():
            if book.approved:
                books_count += book.count
                price += book.cost1 * book.count
        r.books_count = books_count
        r.price = price

    return render(
        request,
        'orders/manage-requests.html',
        {
            'requests': requests,
            'pages': {
                'current': page,
                'total': total_count
            },
            'admin': request.user.has_perm('books.change_purchaserequest'),
            'states': (
                (RequestFlow.STATE_NEW, u'Новая'),
                (RequestFlow.STATE_ORDERED, u'Заказана'),
                (RequestFlow.STATE_DELIVERED, u'Доставлено'))
        }
    )


@permission_required('books.change_purchaserequest')
def change_request_state(request):
    try:
        purchase_request = PurchaseRequest.objects.get(pk=request.REQUEST['request_id'])
        purchase_request.state = request.REQUEST['state']
        purchase_request.save()
        if request.REQUEST['state'] == RequestFlow.STATE_DELIVERED:
            if purchase_request.delivery_office_id in MOSCOW_OFFICES_IDS:
                office = 'moscow'
            elif purchase_request.delivery_office_id in SPB_OFFICES_IDS:
                office = 'spb'
            else:
                return response_json({'result': 'ok'})

            is_male = purchase_request.customer.gender == u'M'
            books = filter(lambda x: x.approved, purchase_request.requested_books.all())
            body = render_to_string(
                'orders/order-delivered-mail.html',
                {
                    'order': purchase_request,
                    'male': is_male,
                    'multiple': len(books) > 1,
                    'books': books,
                    'office': office,
                }
            )
            email_from = settings.SERVER_EMAIL
            copy = [settings.RETURN_BOOK_FROM]
            to = [purchase_request.customer.email]
            subj = u'Заказ №%d доставлен' % (purchase_request.id)
            if email_from and to:
                msg = EmailMessage(subj, body, email_from, to, cc=copy)
                msg.content_subtype = "html"
                msg.send()
        return response_json({'result': 'ok'})
    except:
        traceback.print_exc()
        return response_json({'result': 'error'})


@permission_required('books.change_purchaserequest')
def change_request_approve(request):
    try:
        book = PurchaseRequestOneBook.objects.get(pk=request.REQUEST['book_id'])
        if request.REQUEST['action'] == 'positive':
            book.approved = True
        if request.REQUEST['action'] == 'negative':
            book.approved = False
        book.save()
        r = book.request
        price = 0
        for b in r.requested_books.all():
            if b.approved:
                price += b.cost1 * b.count
        return response_json({'result': 'ok', 'price': price})
    except:
        traceback.print_exc()
        return response_json({'result': 'error'})



