# -*- coding: utf-8 -*-
from django.conf import settings
from django.shortcuts import render_to_response
from django.contrib.auth.decorators import login_required
from django.template import RequestContext

from libra.books.models import TagCategory
from libra.tagging.models import Tag


def make_tags_columns(max_rows=None):
    '''
    Формирует списки тегов, разделенные на Категории.
    Можно показывать только часть тегов
    '''

    tag_cats = list(TagCategory.objects.select_related().all())
    grouped_tags = []
    for tag_cat in tag_cats:
        tags_rest = None
        if max_rows:
            all_tags = tag_cat.tags.all()
            tags = all_tags[:max_rows]
            # Сколько тегов останется не показано
            tags_rest = len(all_tags) - max_rows
            if tags_rest < 1:
                tags_rest = None
        else:
            tags = tag_cat.tags.all()

        if tag_cats.index(tag_cat) == 0:
            group1 = []
            group2 = []
            for tag in tags:
                fc = tag.name[:1].lower()
                if fc.isdigit() or (fc >= 'a' and fc <= 'z'):
                    group1.append(tag)
                else:
                    group2.append(tag)
            grouped_tags.append((tag_cat.name, group1, tags_rest))
            grouped_tags.append((tag_cat.name, group2, tags_rest))
        else:
            grouped_tags.append((tag_cat.name, tags, tags_rest))

    return grouped_tags


def make_plain_tags_list(grouped_tags):
    u' разворачивает сгруппированные теги в плоский список тегов '
    tags_list = []
    for none, tags, none2 in grouped_tags:
        tags_list = tags_list + list(tags)

    return tags_list


def make_tags_rubricator():
    '''
    Формирует несколько списков рубрик: сгруппированный по категориям,
    рубрики без категории, и плоский список всех рубрик
    '''

    tags_groups = make_tags_columns()

    # рубрики, которым сопоставлены какие-то книги
    used_tags_plain = make_plain_tags_list(tags_groups)
    # IDs используемых рубрик
    cat_tags_ids = [ct.id for ct in used_tags_plain]
    all_tags_plain = Tag.objects.all().order_by('name')  # Вообще все рубрики
    # неиспользуемые рубрики (пойдут в группу "Прочие")
    orphaned_tags = all_tags_plain.exclude(id__in=cat_tags_ids)

    return {
        'tags_groups': tags_groups,
        'orphaned_tags': orphaned_tags,
        'all_tags_plain': all_tags_plain
    }


@login_required(redirect_field_name=settings.REDIRECT_FIELD_NAME)
def list_tags(request):
    u' вьюха страницы "Рубрикатор" '
    office_id = request.GET.get('office', None)
    if office_id:
        office_s = '&office=%s' % office_id
    else:
        office_s = ''

    tags_rubricator = make_tags_rubricator()
    tags_groups = tags_rubricator['tags_groups']
    orphaned_tags = tags_rubricator['orphaned_tags']

    return render_to_response(
        'view-tags.html',
        {
            'tags_groups': tags_groups,
            'orphaned_tags': orphaned_tags,
            'office_s': office_s
        },
        context_instance=RequestContext(request)
    )
