# coding: utf-8
from django.conf import settings
from django.shortcuts import render_to_response
from django.template.context import RequestContext
from django.template.loader import render_to_string

from libra.shortcuts import response_json
from libra.books.models import BookItem
from libra.kiosks.models import Kiosk
from libra.rfid import Rfid
from django_intranet_stuff.models import Staff, Office

import logging

rfid_log = logging.getLogger('rfid_log')

def kiosk_log_live(kiosk):
    u' Обновляет дату-время последнего обращения к киоску и логгирует действия '
    if kiosk:
        kiosk.save()


def get_kiosk(kiosk_id):
    u' Возвращает Киоск по его ID '
    try:
        return Kiosk.objects.get(id=int(kiosk_id))
    except Kiosk.DoesNotExist:
        pass

def get_kiosk_and_log_live(kiosk_id):
    u' Возвращает инстанс Киоска по id и логгирует обращение к нему '
    kiosk = get_kiosk(kiosk_id)
    if kiosk:
        kiosk_log_live(kiosk)
        return kiosk
    else:
        return None

def page(request, kiosk_id):
    u' Возвращает html-шаблон базовой страницы киоска '
    get_kiosk_and_log_live(kiosk_id)
    request.kiosk_id = kiosk_id
    return render_to_response('page.html', {"kiosk_id": kiosk_id}, context_instance=RequestContext(request))


def ping_kiosk(request, kiosk_id):
    u' Пинговалка киоска и проброс принудительной перезагрузки (если задана) '
    kiosk = get_kiosk_and_log_live(kiosk_id)
    actions = {
        True: 'reload',
        False: '',
    }

    action = actions[kiosk.force_reload]
    if kiosk.force_reload:
        kiosk.force_reload = False
        kiosk.save()

    return response_json({
        'action': action,
    })

def start(request, kiosk_id):
    u' Возвращает Стартовый Экран в виде HTML-in-JSON '
    get_kiosk_and_log_live(kiosk_id)
    result = {
        'html': render_to_string('start.html', context_instance=RequestContext(request)),
    }
    return response_json(result)


def basket(request, kiosk_id):
    u' Возвращает Экран Корзины в виде HTML-in-JSON '
    basket_mode = request.GET.get('mode', None)
    get_kiosk_and_log_live(kiosk_id)

    result = {
        'html': render_to_string('basket.html', {},
                                 context_instance=RequestContext(request)),
    }
    return response_json(result)


def rfid(request, kiosk_id):
    u' Возвращает объект по заданному RFID (экземпляр книги или человека) '
    kiosk = get_kiosk_and_log_live(kiosk_id)

    if not kiosk:
        return response_json({'result': 'error', 'msg': 'kiosk id not valid.'})

    rfid_string = request.GET.get('rfid', None)
    screen = request.GET.get('screen', None)

    rfid_log.info(u'RFID:: %s / screen "%s" / kiosk %d' % (rfid_string, screen, kiosk.id))

    result = {
        'result': 'ok',
    }
    if rfid_string and screen:
        #rfid_number = Rfid.hex2decimal(rfid_string) # мы всегда ожидаем тут ввод в HEX от ридера
        rfid_number = (int(Rfid.hex2decimal(rfid_string)) & 0x7fffffff) % 1000000000 # мы всегда ожидаем тут ввод в HEX от ридера
        if rfid_number:
            result['rfid_number'] = rfid_number
            # сначала поищем номер среди книг
            try:
                bi = BookItem.objects.get(rfid=rfid_number)
                if bi.book.main_cover:
                    book_cover_url = bi.book.main_cover.img_large.url
                else:
                    book_cover_url = None
                result.update({
                    'type'           : 'book',
                    'bookitem_id'    : bi.id,
                    'book_cover_url' : book_cover_url,
                    'book_title'     : bi.book.title,
                    'book_authors'   : bi.book.authors_string,
                    'case'           : bi.case,
                    'shelf'          : bi.shelf,
                    'html'           : render_to_string('basket-book-item.html',{'bi': bi})
                })
            except BookItem.DoesNotExist:
                # если не нашли, ищем среди людей
                rfid = Rfid(settings.RFID_PROVIDER_URL, settings.USER_AGENT) # спец. объект для резолвинга rfid в логин сотрудника
                rfid_found = rfid.resolve(rfid_number)
                if rfid_found:
                    if rfid.is_employee():
                        result.update({
                            'type':'user',
                            'login': rfid.login,
                            'active': rfid.is_active(),
                        })
                    else:
                        result.update({
                            'result':'error',
                            'message': u'Ой, по вашему пропуску мы не можем работать с книгами :('
                        })
                else:
                    result.update({
                        'result':'error',
                        'message': u'Вы поднесли что-то неизвестное нам :('
                    })
        else:
            result.update({'result':'error'})

    else:
        result.update({'result':'error'})

    rfid_log.info(u'RFID:: %s / response: "%s" / kiosk %d' % (rfid_string, result, kiosk.id))
    return response_json(result)


def basket_action(request, kiosk_id):
    u' Вьюха, проводящая операцию по перемещению книг из корзины к сотруднику или в библиотеку '
    if request.method != 'POST':
        return response_json({'result':'error', 'message':'invalid kioskID provided'})

    kiosk = get_kiosk_and_log_live(kiosk_id)

    if not kiosk:
        return response_json({'result':'error', 'message':'invalid kioskID provided'})

    # должен быть указан допустимый режим корзины: взять/вернуть книги
    mode = request.POST.get('mode', None)
    if mode not in ('take', 'restore'):
        return response_json({'result':'error', 'message':'invalid mode provided'})

    # в списке книг должна быть хотя бы 1 книга
    bookitems_id = request.POST.getlist('books')    # получим список ID экз. книг в виде строк
    if len(bookitems_id) < 1:
        return response_json({'result':'error', 'message':'no books provided'})

    # должен быть передан логин сотрудника
    staff_login = request.POST.get('login', '').strip()
    if not staff_login:
        return response_json({'result':'error', 'message':'invalid staff login provided'})

    try:
        staff = Staff.objects.get(login_ld = staff_login)
    except Staff.DoesNotExist:
        return response_json({'result':'error', 'message':'cant find staff for login provided'})

    result = {
        'result': 'ok',
    }

    success_count = 0
    for bid in bookitems_id:
        try:
            bi = BookItem.objects.get(id=bid)
        except BookItem.DoesNotExist:
            continue # чего делать когда не нашли книжку по ID?

        if mode == 'take':
            bi.move_to_user(staff)      # передадим книжку сотруднику
        elif mode == 'restore':
            bi.move_to_library(kiosk.office, staff)     # вернем книгу в библиотеку

        success_count += 1

    result.update({
        'success_count': success_count,
        'has_problem': True if len(bookitems_id) != success_count else False,
    })

    return response_json(result)

