package api

import (
	"errors"
	"net/http"
	"strconv"
	"strings"
)

type paramsValidationError struct {
	fieldErrors map[string][]error
}

func (p *paramsValidationError) Error() string {
	// return errors details in json format

	if p.fieldErrors == nil {
		return "{}"
	}

	var b strings.Builder
	var bufLen int

	b.WriteByte('{')

	for field, fieldErrors := range p.fieldErrors {
		if bufLen != 0 {
			b.WriteByte(',')
			bufLen = 0
		}

		errors := make([]string, len(fieldErrors))
		for i, err := range fieldErrors {
			if i != 0 {
				bufLen += 1
			}
			errors := append(errors, err.Error())
			bufLen += len(errors[i]) + 2
		}

		b.Grow(bufLen + len(field) + 5)
		b.WriteByte('"')
		b.WriteString(field)
		b.WriteString("\":[")

		for i, err := range errors {
			if i != 0 {
				b.WriteByte(',')
			}
			b.WriteByte('"')
			b.WriteString(err)
			b.WriteByte('"')
		}
	}

	b.WriteByte('}')

	return b.String()
}

func (p *paramsValidationError) addFieldErrors(field string, errs ...error) {
	if p.fieldErrors == nil {
		p.fieldErrors = make(map[string][]error)
	}

	_, ok := p.fieldErrors[field]
	if !ok {
		p.fieldErrors[field] = make([]error, len(errs))
	}
	p.fieldErrors[field] = append(p.fieldErrors[field], errs...)
}

type params struct {
	fields map[string]int
	sort   map[string]int
	limit  int
	page   int
	query  map[string]interface{}
}

func newParams(r *http.Request) (params, error) {
	s := params{query: make(map[string]interface{})}

	paramsError := &paramsValidationError{}
	var err error
	var errs []error

	err = r.ParseForm()
	if err != nil {
		paramsError.addFieldErrors("_global", err)
		return s, paramsError
	}

	params := make(map[string]string)
	for k, v := range r.Form {
		len := len(v)
		if len != 0 {
			params[k] = v[len-1]
		}
	}

	if pageParam, ok := params["_page"]; ok {
		s.page, err = parsePositiveInteger(pageParam)
		if err != nil {
			paramsError.addFieldErrors("_page", err)
		}
	} else {
		s.page = 1
	}

	if limitParam, ok := params["_limit"]; ok {
		s.limit, err = parsePositiveInteger(limitParam)
		if err != nil {
			paramsError.addFieldErrors("_limit", err)
		}
	} else {
		s.limit = 50
	}

	if sortParam, ok := params["_sort"]; ok {
		s.sort, errs = parseArrayWithFields(sortParam)
		if err != nil {
			paramsError.addFieldErrors("_sort", errs...)
		}
	} else {
		s.sort = map[string]int{"id": 1}
	}

	if fieldsParam, ok := params["_fields"]; ok {
		s.fields, errs = parseArrayWithFields(fieldsParam)
		if err != nil {
			paramsError.addFieldErrors("_fields", errs...)
		}
	} else {
		s.fields = map[string]int{"id": 1}
	}

	for field, param := range params {
		if field[0] == '_' {
			continue
		}
		s.query[field], err = parseQueryField(field, param)
		if err != nil {
			paramsError.addFieldErrors("_query", err)
		}
	}

	if len(paramsError.fieldErrors) != 0 {
		return s, paramsError
	}
	return s, nil
}

func parsePositiveInteger(value string) (int, error) {
	integer, err := strconv.Atoi(value)
	if err != nil {
		return 0, err
	}
	if integer < 1 {
		return 0, errors.New("should be number greater than or equal to 1")
	}
	return integer, nil
}

func parseArrayWithFields(value string) (map[string]int, []error) {
	errs := make([]error, 0)
	fields := strings.Split(value, ",")
	for _, field := range fields {
		if len(field) == 0 {
			errs = append(errs, errors.New("unknown field ``"))
		}
		// TODO: check field existence
	}
	if len(errs) != 0 {
		return nil, errs
	}
	result := make(map[string]int)
	for _, field := range fields {
		if field[0] == '-' {
			result[field[1:]] = -1 // for _sort
		} else {
			result[field] = 1
		}
	}

	return result, nil
}

func parseQueryField(field, value string) (interface{}, error) {
	return "kek", nil
}
