package api

import (
	"context"
	"encoding/json"
	"net/http"
	"net/url"
	"strconv"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/intranet/legacy/staff-api/internal/database"
)

func GetRoutes(db database.Client) http.Handler {
	r := chi.NewRouter()

	r.Use(middleware.StripSlashes)

	r.Get("/positions", getResourceHandler(db, "position"))

	return r
}

func writeError(w http.ResponseWriter, code int, message string, details error) {
	w.WriteHeader(code)
	w.Header().Set("Content-Type", "application/json")

	response := struct {
		ErrorMessage string `json:"error_message"`
		Details      string `json:"details"`
	}{ErrorMessage: message, Details: details.Error()}
	raw, _ := json.Marshal(&response)
	_, _ = w.Write(raw)
}

func getLinks(r *http.Request, page, maxPage int) (next, prev, first, last string) {
	url := new(url.URL)
	*url = *r.URL

	query := url.Query()

	if maxPage == 0 {
		return
	}

	if page != 1 {
		query.Set("_page", "1")
		url.RawQuery = query.Encode()
		first = url.String()

		query.Set("_page", strconv.Itoa(page-1))
		url.RawQuery = query.Encode()
		prev = url.String()
	}

	if page != maxPage {
		query.Set("_page", strconv.Itoa(maxPage))
		url.RawQuery = query.Encode()
		last = url.String()

		query.Set("_page", strconv.Itoa(page+1))
		url.RawQuery = query.Encode()
		next = url.String()
	}

	return
}

func getCollectionForResourse(db database.Client, resource string) (string, error) {
	var result struct{ Prefixes struct{ Read string } }
	err := db.GetOne(context.TODO(), "state", nil, map[string]int{"prefixes.read": 1}, map[string]int{"started_at": -1}, &result)
	if err != nil {
		return "", err
	}
	return result.Prefixes.Read + resource, nil
}

func getResourceHandler(db database.Client, resourse string) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		w.Header().Set("Content-Type", "application/json")

		params, err := newParams(r)
		if err != nil {
			writeError(w, 400, "Parameters validation error", err)
			return
		}

		collection, err := getCollectionForResourse(db, resourse)
		if err != nil {
			writeError(w, 500, "Unable to get read collection", err)
			return
		}

		countChan := make(chan interface{}, 1)

		go func() {
			count, err := db.Count(r.Context(), collection, params.query)
			if err != nil {
				countChan <- err
			} else {
				countChan <- count
			}
		}()

		result := make([]Position, 0) // FIXME
		err = db.Get(r.Context(), collection, params.query, params.fields, params.sort, params.limit, params.page, &result)
		if err != nil {
			writeError(w, 500, "Search failed", err)
			return
		}

		countChanValue := <-countChan
		if err, ok := countChanValue.(error); ok {
			writeError(w, 500, "Count failed", err)
			return
		}
		count, _ := countChanValue.(int)

		response := struct {
			Links struct {
				Next  string `json:"next,omitempty"`
				Prev  string `json:"prev,omitempty"`
				First string `json:"first,omitempty"`
				Last  string `json:"last,omitempty"`
			} `json:"links"`
			Page   int         `json:"page"`
			Limit  int         `json:"limit"`
			Result interface{} `json:"result"`
			Total  int         `json:"total"`
			Pages  int         `json:"pages"`
		}{
			Page:   params.page,
			Limit:  params.limit,
			Result: result,
			Total:  count,
			Pages:  count / params.limit,
		}

		if count%params.limit != 0 {
			response.Pages += 1
		}

		response.Links.Next, response.Links.Prev, response.Links.First, response.Links.Last = getLinks(r, response.Page, response.Pages)

		raw, err := json.Marshal(response)
		if err != nil {
			writeError(w, 500, "Unable to encode json from mongo", err)
		}

		w.WriteHeader(200)
		_, _ = w.Write(raw)
	}
}
