package idm

import (
	"encoding/json"
	"errors"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/blackbox"

	"a.yandex-team.ru/intranet/legacy/staff-api/internal/api"
	"a.yandex-team.ru/intranet/legacy/staff-api/internal/database"
)

type Response struct {
	Code int `json:"code"`

	Users []User `json:"users,omitempty"`
	Roles *Group `json:"roles,omitempty"`

	Warning string `json:"warning,omitempty"`
	Error   string `json:"error,omitempty"`
	Fatal   string `json:"fatal,omitempty"`
}

type Role map[string]string

type User struct {
	Login       string `json:"login"`
	Roles       []Role `json:"roles"`
	SubjectType string `json:"subject_type"`
}

type Group struct {
	Slug   string            `json:"slug"`
	Name   string            `json:"name"`
	Values map[string]string `json:"values"`
}

const (
	CodeOK    = 0
	CodeNotOK = 1

	collection = "roles"
	rolesField = "roles"

	rolesSlug = "role"
)

func parseParams(v url.Values) (login string, uid blackbox.ID, role string, err error) {
	// return login, uid for user and login == str(uid) == str(tvm_id) for tvm app
	if logins, ok := v["login"]; ok {
		login = logins[0]
	} else {
		err = errors.New("login: parameter not provided")
		return
	}

	if uids, ok := v["uid"]; ok {
		uid, err = strconv.ParseUint(uids[0], 10, 0)
		if err != nil {
			err = errors.New("uid: unable to parse (not uint)")
			return
		}
	} else {
		if subject, ok := v["subject_type"]; ok && subject[0] == "tvm_app" {
			uid, err = strconv.ParseUint(login, 10, 0)
			if err != nil {
				err = errors.New("uid: login is not uint while subject is tmv_app")
				return
			}
		} else {
			err = errors.New("uid: parameter not provided")
			return
		}
	}

	if roles, ok := v["role"]; ok {
		rawRole := roles[0]
		parsedRole := make(map[string]string)
		err = json.Unmarshal([]byte(rawRole), &parsedRole)
		if err != nil {
			err = errors.New("role: unable to parse")
			return
		}
		role, ok = parsedRole[rolesSlug]
		if !ok {
			err = errors.New("role: slug not found")
			return
		}
	} else {
		err = errors.New("role: parameter not provided")
		return
	}

	return
}

func isNumber(value string) bool {
	_, err := strconv.ParseInt(value, 10, 64)
	return err == nil
}

func writeResponse(w http.ResponseWriter, logger log.Logger, Response Response) {
	response, _ := json.Marshal(Response)
	w.Header().Set("Content-Type", "application/json")
	_, err := w.Write(response)
	if err != nil {
		logger.Warn("error trying to write response", log.Error(err))
	}
}

func getInfoHandler(logger log.Logger) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		writeResponse(w, logger, Response{
			Code: CodeOK,
			Roles: &Group{
				Slug:   rolesSlug,
				Name:   "Роли",
				Values: api.RolesSet,
			},
		})
	})
}

func getAllRolesHandler(logger log.Logger, db database.Client) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		documents := make([]struct {
			Login string
			Roles []string
		}, 0)

		if err := db.GetAll(r.Context(), collection, &documents); err != nil {
			writeResponse(w, logger, Response{
				Code:  CodeNotOK,
				Error: err.Error(),
			})
			return
		}

		users := make([]User, len(documents))
		for i, document := range documents {
			users[i].Login = document.Login

			if isNumber(document.Login) {
				users[i].SubjectType = "tvm_app"
			} else {
				users[i].SubjectType = "user"
			}

			users[i].Roles = make([]Role, len(document.Roles))
			for j, role := range document.Roles {
				users[i].Roles[j] = Role{rolesSlug: role}
			}
		}

		writeResponse(w, logger, Response{
			Code:  CodeOK,
			Users: users,
		})
	})
}

func getAddRoleHandler(logger log.Logger, db database.Client) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		Response := Response{}

		body, err := ioutil.ReadAll(r.Body)
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		postParams, err := url.ParseQuery(string(body))
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		login, uid, role, err := parseParams(postParams)
		if err != nil {
			Response.Code = CodeNotOK
			Response.Fatal = err.Error()

			writeResponse(w, logger, Response)
			return
		}

		filter := map[string]interface{}{
			"login": login,
			"uid":   uid,
		}

		err = db.AddItemToArray(r.Context(), collection, rolesField, filter, role)
		if err != nil {
			Response.Code = CodeNotOK
			Response.Error = err.Error()
			writeResponse(w, logger, Response)
			return
		}

		Response.Code = CodeOK
		writeResponse(w, logger, Response)
	})
}

func getRemoveRoleHandler(logger log.Logger, db database.Client) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		Response := Response{}

		body, err := ioutil.ReadAll(r.Body)
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		postParams, err := url.ParseQuery(string(body))
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		login, uid, role, err := parseParams(postParams)
		if err != nil {
			Response.Code = CodeNotOK
			Response.Fatal = err.Error()

			writeResponse(w, logger, Response)
			return
		}

		filter := map[string]interface{}{
			"login": login,
			"uid":   uid,
		}

		err = db.PullItemFromArray(r.Context(), collection, rolesField, filter, role)
		if err != nil {
			Response.Code = CodeNotOK
			Response.Error = err.Error()

			writeResponse(w, logger, Response)
			return
		}

		Response.Code = CodeOK
		writeResponse(w, logger, Response)
	})
}
